package ru.yandex.chemodan.app.psbilling.core.dao.promos;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoStatusType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.UserPromoEntity;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class UserPromoDaoImpl extends AbstractDaoImpl<UserPromoEntity> implements UserPromoDao {
    public UserPromoDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    private static final String USER_PROMOS_TABLE = "user_promos";
    private static final String PROMO_PRODUCT_LINES_TABLE = "promo_product_lines";

    @Override
    public String getTableName() {
        return USER_PROMOS_TABLE;
    }

    @Override
    public UserPromoEntity createOrUpdate(InsertData data) {
        if (data.getToDate().isPresent() && data.getFromDate().isAfter(data.getToDate().get())) {
            throw new BadRequestException("to_date can't be before from_date");
        }

        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("promo_template_id", data.getPromoTemplateId());
        params.put("uid", data.getUid());
        params.put("from_date", data.getFromDate());
        params.put("to_date", data.getToDate().getOrNull());
        params.put("status", data.getPromoStatusType());
        params.put("now", now);

        return jdbcTemplate.queryForOption(
                "insert into " + USER_PROMOS_TABLE + " as t (" +
                        "    created_at," +
                        "    updated_at," +
                        "    promo_template_id," +
                        "    uid," +
                        "    from_date, " +
                        "    to_date, " +
                        "    status) " +
                        "values(" +
                        "    :now, " +
                        "    :now, " +
                        "    :promo_template_id, " +
                        "    :uid, " +
                        "    :from_date, " +
                        "    :to_date, " +
                        "    :status::user_promos_status) " +
                        "ON CONFLICT (promo_template_id, uid) DO UPDATE " +
                        "SET from_date = :from_date, " +
                        "    to_date = :to_date, " +
                        "    updated_at = :now, " +
                        "    status = :status::user_promos_status " +
                        "RETURNING *",
                (rs, num) -> parseRow(rs), params)
                .getOrThrow(() ->
                        new BadRequestException("can't update existing user promo because it is already used"));
    }

    @Override
    public ListF<UserPromoEntity> findUserPromos(PassportUid uid) {
        return jdbcTemplate.query("select * from " + USER_PROMOS_TABLE + " where uid = ?",
                (rs, rowNum) -> parseRow(rs), uid.toString());
    }

    @Override
    public ListF<UserPromoEntity> findUserPromos(PassportUid uid, ListF<UUID> productLines) {
        if (productLines.isEmpty()) {
            return Cf.list();
        }

        MapF<String, Object> params = Cf.hashMap();
        params.put("productLineIds", productLines);
        params.put("uid", uid.toString());

        return jdbcTemplate.query(
                "select distinct pt.* from " + USER_PROMOS_TABLE + " pt " +
                        "join " + PROMO_PRODUCT_LINES_TABLE + " pl on pt.promo_template_id = pl.promo_template_id " +
                        "where pt.uid = :uid and pl.product_line_id in (:productLineIds)",
                (rs, num) -> parseRow(rs), params);
    }

    @Override
    public Option<UserPromoEntity> findUserPromo(PassportUid uid, UUID promoTemplateId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("promoTemplateId", promoTemplateId);
        params.put("uid", uid.toString());

        return jdbcTemplate.queryForOption(
                "select pt.* from " + USER_PROMOS_TABLE + " pt " +
                        "where pt.uid = :uid and pt.promo_template_id = :promoTemplateId",
                (rs, num) -> parseRow(rs), params);
    }

    @Override
    public UserPromoEntity setStatus(UUID id, PromoStatusType status) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", id);
        params.put("status", status);
        params.put("now", Instant.now());

        return jdbcTemplate.queryForOption(
                "update " + USER_PROMOS_TABLE + " set status = :status::user_promos_status, " +
                        "updated_at = :now " +
                        "Where id = :id " +
                        "RETURNING *",
                (rs, num) -> parseRow(rs), params)
                .getOrThrow("user promo with id=" + id + " not found");
    }

    @Override
    public UserPromoEntity parseRow(ResultSet rs) throws SQLException {
        return new UserPromoEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                UUID.fromString(rs.getString("promo_template_id")),
                PassportUid.cons(Long.parseLong(rs.getString("uid"))),
                new Instant(rs.getTimestamp("from_date")),
                Option.ofNullable(rs.getTimestamp("to_date")).map(Instant::new),
                PromoStatusType.R.fromValue(rs.getString("status")),
                new Instant(rs.getTimestamp("updated_at")));
    }
}
