package ru.yandex.chemodan.app.psbilling.core.dao.promos.group;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.intellij.lang.annotations.Language;
import org.joda.time.Instant;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.CreatedOrExistResult;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.group.GroupPromoEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.group.GroupPromoStatusType;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class GroupPromoDaoImpl extends AbstractDaoImpl<GroupPromoEntity> implements GroupPromoDao {
    public GroupPromoDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "group_promos";
    }

    @Override
    public CreatedOrExistResult<GroupPromoEntity> createIfNotExist(InsertData data) {
        Instant now = Instant.now();

        MapF<String, Object> params = Cf.hashMap();
        params.put("promo_template_id", data.getPromoTemplateId());
        params.put("group_id", data.getGroupId());
        params.put("from_date", data.getFromDate());
        params.put("to_date", data.getToDate().orElseGet(() -> null));
        params.put("status", data.getStatus());
        params.put("now", now);

        @Language("SQL")
        String sql = "WITH ins AS (" +
                "INSERT INTO " + getTableName() +
                " (created_at, updated_at, status, promo_template_id, group_id, from_date, to_date) " +
                " VALUES " +
                " (:now, :now, :status::group_promo_status, :promo_template_id, :group_id, :from_date," +
                ":to_date) " +
                " ON CONFLICT (promo_template_id, group_id) DO NOTHING RETURNING *" +
                " )" +
                " SELECT *, true as is_create FROM ins" +
                " UNION ALL" +
                " SELECT *, false as is_create FROM " + getTableName() +
                " WHERE promo_template_id = :promo_template_id and group_id = :group_id";

        return jdbcTemplate.queryForOption(
                        sql,
                        (rs, num) -> CreatedOrExistResult.cons(rs.getBoolean("is_create"), parseRow(rs)),
                        params
                )
                .orElseThrow(() -> new EmptyResultDataAccessException("group promo with" +
                        " promo_template_id=" + data.getPromoTemplateId() +
                        " group_id=" + data.getGroupId() +
                        " not found", 1));
    }

    @Override
    public GroupPromoEntity updateById(UUID id, UpdateData data) {
        Instant now = Instant.now();

        MapF<String, Object> params = Cf.hashMap();
        params.put("id", id);
        params.put("from_date", data.getFromDate());
        params.put("to_date", data.getToDate().orElseGet(() -> null));
        params.put("status", data.getStatus());
        params.put("now", now);

        @Language("SQL") String sql = "UPDATE " + getTableName() + " SET" +
                " status = :status::group_promo_status, updated_at = :now, from_date = :from_date, to_date = :to_date" +
                " WHERE id = :id" +
                " RETURNING *";

        return jdbcTemplate.queryForOption(sql, (rs, num) -> parseRow(rs), params)
                .orElseThrow(() -> new EmptyResultDataAccessException("group promo with id=" + id + " not found", 1));
    }

    @Override
    public ListF<GroupPromoEntity> findByGroupId(UUID groupId) {
        @Language("SQL")
        String sql = "SELECT * FROM " + getTableName() + " WHERE group_id = ?";

        return jdbcTemplate.query(sql, (rs, num) -> parseRow(rs), groupId);
    }

    @Override
    public Option<GroupPromoEntity> findByGroupIdAndPromoTemplateId(UUID groupId, UUID promoTemplateId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("promo_template_id", promoTemplateId);
        params.put("group_id", groupId);

        @Language("SQL")
        String sql = "SELECT * FROM " + getTableName() +
                " WHERE group_id = :group_id AND promo_template_id = :promo_template_id";

        return jdbcTemplate.queryForOption(sql, (rs, num) -> parseRow(rs), params);
    }

    @Override
    public GroupPromoEntity updateStatusById(UUID id, GroupPromoStatusType status) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", id);
        params.put("status", status);
        params.put("now", Instant.now());

        @Language("SQL") String sql = "UPDATE " + getTableName() +
                " SET status = :status::group_promo_status, updated_at = :now" +
                " WHERE id = :id" +
                " RETURNING *";

        return jdbcTemplate.queryForOption(sql, (rs, num) -> parseRow(rs), params)
                .orElseThrow(() -> new EmptyResultDataAccessException("group promo with id=" + id + " not found", 1));
    }

    @Override
    public GroupPromoEntity parseRow(ResultSet rs) throws SQLException {
        return new GroupPromoEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                UUID.fromString(rs.getString("promo_template_id")),
                UUID.fromString(rs.getString("group_id")),
                new Instant(rs.getTimestamp("from_date")),
                Option.ofNullable(rs.getTimestamp("to_date")).map(Instant::new),
                GroupPromoStatusType.R.fromValue(rs.getString("status")),
                new Instant(rs.getTimestamp("updated_at"))
        );
    }
}
