package ru.yandex.chemodan.app.psbilling.core.dao.promos.group;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.intellij.lang.annotations.Language;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.PromoTemplateWithUsedPromo;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationArea;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.group.GroupPromoEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.group.GroupPromoStatusType;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

@RequiredArgsConstructor
public class GroupPromoTemplateDaoImpl implements GroupPromoTemplateDao {
    private final JdbcTemplate3 jdbcTemplate;

    //@formatter:off
    @Language("SQL")
    private final String findPromoTemplateByProductLineSql = "" +
            "select p.id                            as pt_id, " +
            "       p.description                   as pt_description, " +
            "       p.promo_name_tanker_key_id      as pt_promo_name_tanker_key_id, " +
            "       p.from_date                     as pt_from_date, " +
            "       p.to_date                       as pt_to_date, " +
            "       p.application_area              as pt_application_area, " +
            "       p.application_type              as pt_application_type, " +
            "       p.duration                      as pt_duration, " +
            "       p.duration_measurement          as pt_duration_measurement, " +
            "       p.activation_email_template_key as pt_activation_email_template_key, " +
            "       p.created_at                    as pt_created_at, " +
            "       p.code                          as pt_code, " +
            "       p.mobile_background_url         as pt_mobile_background_url, " +
            "       p.mobile_payload                as pt_mobile_payload, " +
            " " +
            "       gp.id                           as gp_id, " +
            "       gp.created_at                   as gp_created_at, " +
            "       gp.updated_at                   as gp_updated_at, " +
            "       gp.status                       as gp_status, " +
            "       gp.promo_template_id            as gp_promo_template_id, " +
            "       gp.group_id                     as gp_group_id, " +
            "       gp.from_date                    as gp_from_date, " +
            "       gp.to_date                      as gp_to_date, " +
            " " +
            "       pl.product_line_id              as product_line_id " +
            "from promo_templates as p " +
            "         join promo_product_lines as pl on p.id = pl.promo_template_id " +
            "         left join group_promos gp on gp.promo_template_id = p.id and gp.group_id = :groupId" +
            " where 1 = 1 " +
            "  and p.from_date <= :now  " +
            "  and p.application_area::text in (:area) " +
            "  and pl.product_line_id in (:productLines) ";
    //@formatter:on


    //@formatter:off
    @Language("SQL")
    private final String getPromoTemplate = "" +
            "select upt.product_line_id as product_line_id, " +
            "       upt.pt_id, " +
            "       upt.pt_description, " +
            "       upt.pt_promo_name_tanker_key_id, " +
            "       upt.pt_from_date, " +
            "       upt.pt_to_date, " +
            "       upt.pt_application_area, " +
            "       upt.pt_application_type, " +
            "       upt.pt_duration, " +
            "       upt.pt_duration_measurement, " +
            "       upt.pt_activation_email_template_key, " +
            "       upt.pt_created_at, " +
            "       upt.pt_code, " +
            "       upt.pt_mobile_background_url, " +
            "       upt.pt_mobile_payload " +
            "from (select pl.product_line_id, " +
            "             p.id                            as pt_id, " +
            "             p.description                   as pt_description, " +
            "             p.promo_name_tanker_key_id      as pt_promo_name_tanker_key_id, " +
            "             p.from_date                     as pt_from_date, " +
            "             p.to_date                       as pt_to_date, " +
            "             p.application_area              as pt_application_area, " +
            "             p.application_type              as pt_application_type, " +
            "             p.duration                      as pt_duration, " +
            "             p.duration_measurement          as pt_duration_measurement, " +
            "             p.activation_email_template_key as pt_activation_email_template_key, " +
            "             p.created_at                    as pt_created_at, " +
            "             p.code                          as pt_code, " +
            "             p.mobile_background_url         as pt_mobile_background_url, " +
            "             p.mobile_payload                as pt_mobile_payload, " +
            "             ROW_NUMBER()                       OVER (PARTITION BY pl.product_line_id " +
            "                 order by " +
            "                     (case p.application_type when 'multiple_time' then 0 ELSE 1 END), " +
            "                     (case p.application_area " +
            "                          when 'global_b2b' then p.to_date " +
            "                          when 'per_group' then gp.to_date " +
            "                     end) NULLS LAST, " +
            "                     p.code)                   AS row_number " +
            "      from promo_templates as p " +
            "               join promo_product_lines as pl on p.id = pl.promo_template_id " +
            "               left join group_promos gp on gp.promo_template_id = p.id and gp.group_id = :groupId " +
            "      where 1 = 1 " +
            "        and p.application_area in ('global_b2b', 'per_group') " +
            "        and ( " +
            "              (p.id = :additionalPromoId and :timePoint between p.from_date and COALESCE(p.to_date, :timePoint)) " +
            "              OR ( " +
            "                          ( " +
            "                              case p.application_area " +
            "                                  when 'global_b2b' then " +
            "                                      ( " +
            "                                          case " +
            "                                              when (p.application_type = 'multiple_time' or cast(:groupId as uuid) is null)" +
            "                                                  then :timePoint between p.from_date and COALESCE(p.to_date, :timePoint) " +
            "                                              when gp.id is not null " +
            "                                                  then (gp.status = 'active' and " +
            "                                                        :timePoint between gp.from_date and COALESCE(gp.to_date, :timePoint)) " +
            "                                              else :timePoint between p.from_date and COALESCE(p.to_date, :timePoint) " +
            "                                              end " +
            "                                          ) " +
            "                                  else true " +
            "                                  end " +
            "                              ) = true " +
            "                      and ( " +
            "                              case p.application_area " +
            "                                  when 'per_group' then " +
            "                                      ( " +
            "                                          case " +
            "                                              when cast(:groupId as uuid) is null " +
            "                                                  then false " +
            "                                              when gp.id is not null " +
            "                                                  then (gp.status = 'active' and " +
            "                                                        :timePoint between gp.from_date and COALESCE(gp.to_date, :timePoint)) " +
            "                                              else false " +
            "                                              end " +
            "                                          ) " +
            "                                  else true " +
            "                                  end) = true " +
            "                  ) " +
            "          )) as upt " +
            "where upt.row_number = 1";
    //@formatter:on

    /**
     * Берем все продуктовые линейки у которых есть промо акции. Так как эта функция для b2b, то нас интересую промо
     * акции только для b2b (global_b2b, per_group)
     * Если промо акция global_b2b, то условия выборки:
     * если промо акция на несколько активаций (multiple_time) или идентификатор группы не задан, то
     * промо акция актуальна на текущую дату
     * если групповая промо акция есть
     * групповая промо акция активна (status = 'active') и актуальна на текущую дату
     * иначе
     * промо акция актуальна на текущую дату
     * Если промо акция per_group, то условия выборки:
     * если группа не задана
     * промо акция не валидна
     * если групповая промо акция есть
     * групповая промо акция активна (status = 'active') и актуальна на текущую дату
     * иначе
     * промо акция не валидна
     * <p>
     * Дополнительно дополняем выгрузку промо акцией из входящих параметров и проверяем её на актуальна на
     * текущую дату.
     * <p>
     * Далее сортируем продуктовые линейки в разрезе промо акции.
     * Сначала показываем на несколько активаций (multiple_time),
     * далее
     * если глобальные (global_b2b), то учитываем время жизни ДО промо акции
     * если для группы (per_group), то учитываем время жизни ДО групповой промо акции
     * и на конец
     * по идентификатору промо акции
     * <p>
     * Нам интересна только одна продуктовая линейка, самая приоритетная по сортировке.
     *
     * @param groupId           - идентификатор группы
     * @param additionalPromoId - идентификатор промо акции
     * @param timePoint         - временная отметка с учетом которой считается запрос
     * @return
     */
    @Override
    public MapF<UUID, PromoTemplateEntity> findPromoTemplateForInTimePoint(
            Option<UUID> groupId,
            Option<UUID> additionalPromoId,
            Option<Instant> timePoint
    ) {
        MapF<String, Object> params = Cf.map(
                "groupId", groupId.orElseGet(() -> null),
                "additionalPromoId", additionalPromoId.orElseGet(() -> null),
                "timePoint", timePoint.orElseGet(Instant::now)
        );
        return jdbcTemplate.query(getPromoTemplate, this::parseRow, params)
                .toMap(tuple -> tuple._1, tuple -> tuple._2);
    }

    @Override
    public MapF<PromoTemplateWithUsedPromo<GroupPromoEntity>, ListF<UUID>> findPromoTemplateByProductLine(
            Group group,
            ListF<UUID> lineIds
    ) {
        if (lineIds.isEmpty()) {
            return Cf.map();
        }

        MapF<String, Object> params = Cf.map(
                "groupId", group.getId(),
                "productLines", lineIds,
                "area", PromoApplicationArea.b2bList,
                "now", Instant.now()
        );

        return jdbcTemplate.query(findPromoTemplateByProductLineSql, this::parsePromoTemplateWithUsed, params)
                .groupByMapValues(t -> t._1, t -> t._2);
    }

    private Tuple2<UUID, PromoTemplateEntity> parseRow(ResultSet rs, int num) throws SQLException {
        PromoTemplateEntity entity = parsePromoTemplate(rs, num);

        UUID productLineId = UUID.fromString(rs.getString("product_line_id"));

        return Tuple2.tuple(productLineId, entity);
    }

    private Tuple2<PromoTemplateWithUsedPromo<GroupPromoEntity>, UUID> parsePromoTemplateWithUsed(ResultSet rs,
                                                                                                  int num) throws SQLException {
        return Tuple2.tuple(
                new PromoTemplateWithUsedPromo<>(
                        parsePromoTemplate(rs, num),
                        parseGroupPromo(rs)
                ),
                UUID.fromString(rs.getString("product_line_id"))
        );
    }

    @SneakyThrows
    private PromoTemplateEntity parsePromoTemplate(ResultSet rs, int num) {
        return new PromoTemplateEntity(
                UUID.fromString(rs.getString("pt_id")),
                new Instant(rs.getTimestamp("pt_created_at")),
                rs.getString("pt_code"),
                rs.getString("pt_description"),
                new Instant(rs.getTimestamp("pt_from_date")),
                Option.ofNullable(rs.getTimestamp("pt_to_date")).map(Instant::new),
                PromoApplicationArea.R.fromValue(rs.getString("pt_application_area")),
                PromoApplicationType.R.fromValue(rs.getString("pt_application_type")),
                CustomPeriod.fromResultSet(rs, "pt_duration_measurement", "pt_duration"),
                Option.ofNullable(rs.getString("pt_activation_email_template_key")),
                Option.ofNullable(rs.getString("pt_promo_name_tanker_key_id")).map(UUID::fromString));
    }

    @SneakyThrows
    private Option<GroupPromoEntity> parseGroupPromo(ResultSet rs) {
        String id = rs.getString("gp_id");

        if (id == null) {
            return Option.empty();
        }

        return Option.of(
                new GroupPromoEntity(
                        UUID.fromString(id),
                        new Instant(rs.getTimestamp("gp_created_at")),
                        UUID.fromString(rs.getString("gp_promo_template_id")),
                        UUID.fromString(rs.getString("gp_group_id")),
                        new Instant(rs.getTimestamp("gp_from_date")),
                        Option.ofNullable(rs.getTimestamp("gp_to_date")).map(Instant::new),
                        GroupPromoStatusType.R.fromValue(rs.getString("gp_status")),
                        new Instant(rs.getTimestamp("gp_updated_at"))
                )
        );
    }
}
