package ru.yandex.chemodan.app.psbilling.core.dao.texts;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.entities.texts.TankerKeyEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.texts.TankerTranslationEntity;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class TankerKeyDaoImpl extends AbstractDaoImpl<TankerKeyEntity> implements TankerKeyDao {

    public TankerKeyDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "tanker_keys";
    }

    @Override
    public TankerKeyEntity parseRow(ResultSet rs) throws SQLException {
        return new TankerKeyEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                rs.getString("project"),
                rs.getString("key_set"),
                rs.getString("key")
        );
    }

    @Override
    public ListF<TankerKeyEntity> findAllKeys() {
        return jdbcTemplate.query("select * from tanker_keys", (rs, num) -> parseRow(rs));
    }

    @Override
    public MapF<UUID, ListF<TankerTranslationEntity>> findAllTranslations() {
        return jdbcTemplate.query("select * from tanker_translations", (rs, num) -> parseTranslation(rs))
                .groupBy(TankerTranslationEntity::getTankerKeyId);
    }

    @Override
    public void mergeTranslations(ListF<TankerTranslationEntity> translations) {
        jdbcTemplate.batchUpdate(
                "insert into tanker_translations(tanker_key_id, locale, translation_text)" +
                        " values (?, ?, ?)" +
                        " on conflict (tanker_key_id, locale)" +
                        "    DO UPDATE" +
                        "    SET translation_text = excluded.translation_text",
                translations.map(t -> new Object[]{t.getTankerKeyId(), t.getLocale(), t.getTranslationText()})
        );
    }

    @Override
    public TankerKeyEntity create(InsertData dataToInsert) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("project", dataToInsert.getProject());
        params.put("keyset", dataToInsert.getKeySet());
        params.put("key", dataToInsert.getKey());
        params.put("now", now);

        return jdbcTemplate.queryForOption("insert into tanker_keys" +
                        " (created_at, project, key_set, key) " +
                        " values (:now, :project, :keyset, :key ) RETURNING *",
                (rs, rowNum) -> parseRow(rs), params).get();
    }

    @Override
    public Tuple2<UUID, ListF<TankerTranslationEntity>> findTranslations(String tankerProject, String tankerKeyset,
            String tankerKey)
    {
        MapF<String, Object> params = Cf.hashMap();
        params.put("project", tankerProject);
        params.put("keyset", tankerKeyset);
        params.put("key", tankerKey);

        return jdbcTemplate.query("select tr.* from tanker_translations tr " +
                        " join tanker_keys k on k.id = tr.tanker_key_id " +
                        " where k.project = :project and k.key_set = :keyset and k.key = :key",
                (rs, num) -> parseTranslation(rs), params)
                .groupBy(TankerTranslationEntity::getTankerKeyId)
                .entries().first();
    }

    @Override
    public void insertIfNotExist(String tankerProject, String tankerKeyset,
                                 String tankerKey) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("project", tankerProject);
        params.put("keyset", tankerKeyset);
        params.put("key", tankerKey);
        params.put("now", now);

        jdbcTemplate.update("insert into tanker_keys" +
                        " (created_at, project, key_set, key) " +
                        " values (:now, :project, :keyset, :key ) " +
                        " on conflict (project, key_set, key) do nothing",
                params);
    }

    @Override
    public boolean exists(String tankerProject, String tankerKeyset, String tankerKey)
    {
        MapF<String, Object> params = Cf.hashMap();
        params.put("project", tankerProject);
        params.put("keyset", tankerKeyset);
        params.put("key", tankerKey);

        return jdbcTemplate.queryForOption("select * from tanker_keys" +
                                " where project = :project and key_set = :keyset and key = :key",
                        (rs, num) -> parseRow(rs), params).isPresent();
    }

    @Override
    public ListF<TankerTranslationEntity> findTranslations(UUID tankerKeyId) {
        return jdbcTemplate.query("select * from tanker_translations where tanker_key_id = ?",
                (rs, num) -> parseTranslation(rs), tankerKeyId);
    }

    @Override
    public MapF<UUID, ListF<TankerTranslationEntity>> findTranslations(ListF<UUID> tankerKeyIds) {
        if (tankerKeyIds.isEmpty()) {
            return Cf.map();
        }
        return jdbcTemplate.query("select * from tanker_translations where tanker_key_id in ( :ids )",
                (rs, num) -> parseTranslation(rs), Cf.map("ids", tankerKeyIds))
                .groupBy(TankerTranslationEntity::getTankerKeyId);
    }

    private static TankerTranslationEntity parseTranslation(ResultSet rs) throws SQLException {
        return new TankerTranslationEntity(
                UUID.fromString(rs.getString("tanker_key_id")),
                rs.getString("locale"),
                rs.getString("translation_text")
        );
    }
}
