package ru.yandex.chemodan.app.psbilling.core.dao.users.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.users.OrderDao;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Order;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderType;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class OrderDaoImpl extends AbstractDaoImpl<Order> implements OrderDao {

    public OrderDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "orders";
    }

    @Override
    public Order parseRow(ResultSet rs) throws SQLException {
        String userServiceId = rs.getString("user_service_id");
        String trustErrorCode = rs.getString("trust_error_code");
        String trustErrorMessage = rs.getString("trust_error_message");
        String packageName = rs.getString("package_name");
        String upgradedOrder = rs.getString("upgraded_order_id_to");

        return new Order(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                UUID.fromString(rs.getString("user_product_price_id")),
                StringUtils.isBlank(userServiceId) ? Option.empty() : Option.of(UUID.fromString(userServiceId)),
                rs.getString("uid"),
                OrderStatus.R.fromValue(rs.getString("status")),
                OrderType.R.fromValue(rs.getString("type")),
                rs.getString("trust_order_id"),
                rs.getInt("subscriptions_count"),
                new Instant(rs.getTimestamp("updated_at")),
                StringUtils.isBlank(trustErrorCode) ? Option.empty() : Option.of(trustErrorCode),
                StringUtils.isBlank(trustErrorMessage) ? Option.empty() : Option.of(trustErrorMessage),
                rs.getInt("trust_service_id"),
                StringUtils.isBlank(packageName) ? Option.empty() : Option.of(packageName),
                StringUtils.isBlank(upgradedOrder) ? Option.empty() : Option.of(UUID.fromString(upgradedOrder)),
                Option.ofNullable(rs.getTimestamp("inapp_synced_at")).map(Instant::new));
    }

    @Override
    public int countInitCreatedBefore(Duration createdBefore) {
        return jdbcTemplate
                .queryForInt("select count(*) from orders where status = ? " +
                        "and created_at < ?", OrderStatus.INIT.value(), Instant.now().minus(createdBefore));
    }

    @Override
    public Order updateOrderPrice(UUID orderId, UUID priceId, int subscriptionsCount) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", Instant.now());
        params.put("price_id", priceId);
        params.put("id", orderId);
        params.put("subs_cnt", subscriptionsCount);

        return jdbcTemplate.query("update orders set user_product_price_id = :price_id, " +
                " subscriptions_count = :subs_cnt, updated_at = :now where id = :id " +
                " returning *", (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public Order updateInappSyncDate(UUID orderId, Instant syncTime) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("sync_time", syncTime);
        params.put("id", orderId);

        return jdbcTemplate.query("update orders set inapp_synced_at = :sync_time " +
                " where id = :id " +
                " returning *", (rs, num) -> parseRow(rs), params).first();
    }

    public ListF<Order> findInappOrderForMigration(Instant minHoldDate, Instant maxPurchaseDate) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("status", OrderStatus.ON_HOLD.value());
        params.put("min_disabled", minHoldDate);
        params.put("max_created", maxPurchaseDate);
        params.put("type", OrderType.INAPP_SUBSCRIPTION.value());
        params.put("service_target", Target.ENABLED.value());

        return jdbcTemplate
                .query("select o.* from orders o " +
                        "join user_services us on o.user_service_id = us.id " +
                        "where o.status = :status " +
                        "and type = :type " +
                        "and us.actual_disabled_at > :min_disabled " +
                        "and o.created_at < :max_created " +
                        "and not exists (" +
                                "select * from user_services " +
                                "where target = :service_target " +
                                "and uid = o.uid) " +
                                "and last_payment_order_id is not null " +
                        "and o.uid not in (select uid::text from inapp_migrations)",
                        (rs, num) -> parseRow(rs), params);
    }

    @Override
    public ListF<UUID> findInStatus(OrderStatus orderStatus, Option<UUID> from, int batch) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("status", orderStatus.value());
        params.put("batch", batch);
        String condition = "";
        if (from.isPresent()) {
            condition = " and id > :from ";
            params.put("from", from.get());
        }

        return jdbcTemplate
                .query("select id from orders where status = :status " + condition + " order by id limit :batch",
                        (rs, num) -> UUID.fromString(rs.getString("id")), params);
    }

    @Override
    public void onSuccessfulOrderPurchase(UUID orderId, Option<UUID> userServiceId, int subscriptionsCount) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", Instant.now());
        params.put("status", OrderStatus.PAID.value());
        params.put("subs_cnt", subscriptionsCount);
        params.put("id", orderId);
        params.put("user_service_id", userServiceId.orElse((UUID) null));

        jdbcTemplate.update("update orders set status = :status, user_service_id = :user_service_id " +
                ", subscriptions_count = :subs_cnt, updated_at = :now where id = :id ", params);
    }

    public void onSuccessfulOrderResume(UUID orderId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", Instant.now());
        params.put("status", OrderStatus.INIT.value());
        params.put("id", orderId);

        jdbcTemplate.update("update orders set status = :status " +
                ", updated_at = :now where id = :id ", params);
    }

    @Override
    public Order holdOrder(UUID orderId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", Instant.now());
        params.put("status", OrderStatus.ON_HOLD.value());
        params.put("id", orderId);

        return jdbcTemplate.query("update orders set status = :status, updated_at = :now where id = :id returning *",
                (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public Order unHoldOrder(UUID orderId) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", Instant.now());
        params.put("hold_status", OrderStatus.ON_HOLD.value());
        params.put("status", OrderStatus.PAID.value());
        params.put("id", orderId);

        return jdbcTemplate.query("update orders set status = :status, updated_at = :now " +
                        "where id = :id and status = :hold_status returning *",
                (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public Order lock(UUID orderId) {
        return jdbcTemplate
                .queryForObject("select * from orders where id = ? for update", (rs, num) -> parseRow(rs), orderId);
    }

    @Override
    public void updateSubscriptionsCount(UUID orderId, int subscriptionsCount) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", Instant.now());
        params.put("subs_count", subscriptionsCount);
        params.put("id", orderId);

        jdbcTemplate.update("update orders set subscriptions_count = :subs_count, updated_at = :now where id = :id ",
                params);
    }

    @Override
    public void writeErrorStatus(UUID orderId, String trustErrorCode, String trustErrorMessage) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", Instant.now());
        params.put("status", OrderStatus.ERROR.value());
        params.put("trust_error_code", trustErrorCode);
        params.put("trust_error_message", trustErrorMessage);
        params.put("id", orderId);

        jdbcTemplate.update("update orders set status = :status, trust_error_code = :trust_error_code, " +
                "trust_error_message = :trust_error_message, updated_at = :now where id = :id ", params);
    }

    @Override
    public Order changeOrderUid(UUID orderId, String uid) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", Instant.now());
        params.put("uid", uid);
        params.put("id", orderId);

        return jdbcTemplate.query("update orders set uid = :uid, updated_at = :now where id = :id returning *",
                (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public Order createOrUpdate(InsertData dataToInsert) {
        Option<OrderStatus> status = dataToInsert.getStatus() == null ? Option.empty() : dataToInsert.getStatus();
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", now);
        params.put("user_product_price_id", dataToInsert.getUserProductPriceId());
        params.put("status", status.orElse(OrderStatus.INIT).value());
        params.put("uid", dataToInsert.getUid());
        params.put("type", dataToInsert.getType().value());
        params.put("trust_order_id", dataToInsert.getTrustOrderId());
        params.put("trust_service_id", dataToInsert.getTrustServiceId());
        params.put("package_name",
                dataToInsert.getPackageName() == null ? null : dataToInsert.getPackageName().orElse((String) null));
        params.put("inapp_synced_at", dataToInsert.getInappSynchronizationDate().orElse((Instant) null));


        return jdbcTemplate.query(
                "insert into orders(user_product_price_id,user_service_id,status,uid,type,trust_order_id, " +
                        "trust_error_message,trust_error_code,created_at,updated_at, trust_service_id, package_name, " +
                        "inapp_synced_at) " +
                        "values(:user_product_price_id, null, :status, :uid, :type, :trust_order_id, null, null," +
                        " :now, :now , :trust_service_id, :package_name, :inapp_synced_at)" +
                        " ON CONFLICT (trust_order_id) DO UPDATE SET updated_at = :now, status = :status " +
                        " RETURNING *",
                (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public Order createIfNotExists(InsertData dataToInsert) {
        Option<OrderStatus> status = dataToInsert.getStatus() == null ? Option.empty() : dataToInsert.getStatus();
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", now);
        params.put("user_product_price_id", dataToInsert.getUserProductPriceId());
        params.put("status", status.orElse(OrderStatus.INIT).value());
        params.put("uid", dataToInsert.getUid());
        params.put("type", dataToInsert.getType().value());
        params.put("trust_order_id", dataToInsert.getTrustOrderId());
        params.put("trust_service_id", dataToInsert.getTrustServiceId());
        params.put("package_name",
                dataToInsert.getPackageName() == null ? null : dataToInsert.getPackageName().orElse((String) null));
        params.put("inapp_synced_at", dataToInsert.getInappSynchronizationDate().orElse((Instant) null));

        return jdbcTemplate.query(
                        "WITH ins AS (" +
                                " insert into orders(user_product_price_id,user_service_id,status,uid,type,trust_order_id," +
                                " trust_error_message,trust_error_code,created_at,updated_at, trust_service_id, package_name," +
                                " inapp_synced_at)" +
                                " values(:user_product_price_id, null, :status, :uid, :type, :trust_order_id, null, null," +
                                " :now, :now , :trust_service_id, :package_name, :inapp_synced_at)" +
                                " ON CONFLICT (trust_order_id) DO NOTHING" +
                                " RETURNING *" +
                                " )" +
                                " SELECT * FROM ins" +
                                " UNION ALL" +
                                " SELECT * FROM orders" +
                                " WHERE trust_order_id = :trust_order_id",
                        (rs, num) -> parseRow(rs),
                        params)
                .first();
    }

    @Override
    public Order create(InsertData dataToInsert) {
        Option<OrderStatus> status = dataToInsert.getStatus() == null ? Option.empty() : dataToInsert.getStatus();
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", now);
        params.put("user_product_price_id", dataToInsert.getUserProductPriceId());
        params.put("status", status.orElse(OrderStatus.INIT).value());
        params.put("uid", dataToInsert.getUid());
        params.put("type", dataToInsert.getType().value());
        params.put("trust_order_id", dataToInsert.getTrustOrderId());
        params.put("trust_service_id", dataToInsert.getTrustServiceId());
        params.put("package_name",
                dataToInsert.getPackageName() == null ? null : dataToInsert.getPackageName().orElse((String) null));
        params.put("inapp_synced_at", dataToInsert.getInappSynchronizationDate().orElse((Instant) null));

        return jdbcTemplate.query(
                        "insert into orders(user_product_price_id,user_service_id,status,uid,type,trust_order_id," +
                                " trust_error_message,trust_error_code,created_at,updated_at, trust_service_id, package_name," +
                                " inapp_synced_at)" +
                                " values(:user_product_price_id, null, :status, :uid, :type, :trust_order_id, null, null," +
                                " :now, :now , :trust_service_id, :package_name, :inapp_synced_at)" +
                                " RETURNING *",
                        (rs, num) -> parseRow(rs),
                        params)
                .first();
    }


    @Override
    public Option<Order> findByTrustOrderId(String trustOrderId) {
        return jdbcTemplate
                .query("select * from orders where trust_order_id = ?", (rs, num) -> parseRow(rs), trustOrderId)
                .firstO();
    }

    @Override
    public ListF<Order> findByServiceId(UUID userServiceId) {
        return jdbcTemplate
                .query("select * from orders where user_service_id = ?", (rs, num) -> parseRow(rs), userServiceId);
    }

    @Override
    public ListF<Order> findByUid(PassportUid uid) {
        return jdbcTemplate
                .query("select * from orders where uid = ?", (rs, num) -> parseRow(rs), uid.toString());
    }

    @Override
    public Order upgradeOrder(UUID orderId, UUID upgradedOrder) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("id", orderId);
        params.put("status", OrderStatus.UPGRADED);
        params.put("upgraded_order_id_to", upgradedOrder);
        params.put("now", Instant.now());

        return jdbcTemplate.query("update orders set status = :status, updated_at = :now, upgraded_order_id_to = " +
                        ":upgraded_order_id_to " +
                        "where id = :id returning *",
                (rs, num) -> parseRow(rs), params).first();
    }
}
