package ru.yandex.chemodan.app.psbilling.core.dao.users.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.users.RefundDao;
import ru.yandex.chemodan.app.psbilling.core.entities.users.Refund;
import ru.yandex.chemodan.app.psbilling.core.entities.users.RefundStatus;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class RefundDaoImpl extends AbstractDaoImpl<Refund> implements RefundDao {

    public RefundDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
    }

    @Override
    public String getTableName() {
        return "refunds";
    }

    @Override
    public Refund parseRow(ResultSet rs) throws SQLException {
        return new Refund(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                new Instant(rs.getTimestamp("updated_at")),
                UUID.fromString(rs.getString("order_id")),
                rs.getString("trust_payment_id"),
                rs.getString("trust_refund_id"),
                RefundStatus.R.fromValue(rs.getString("status"))
        );
    }

    @Override
    public Option<Refund> findByTrustRefundId(String trustRefundId) {
        return jdbcTemplate.queryForOption("select * from refunds where trust_refund_id = ?",
                (rs, rowNum) -> parseRow(rs), trustRefundId);
    }

    @Override
    public Refund create(InsertData insertData) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("trust_payment_id", insertData.getTrustPaymentId());
        params.put("order_id", insertData.getOrderId());
        params.put("status", insertData.getStatus() == null ?
                RefundStatus.INIT.value() : insertData.getStatus().getOrElse(RefundStatus.INIT).value());
        params.put("trust_refund_id", insertData.getTrustRefundId());
        params.put("now", Instant.now());

        return jdbcTemplate.query("insert into refunds (order_id, status, trust_payment_id, " +
                "trust_refund_id, created_at, updated_at ) values " +
                " (:order_id, :status, :trust_payment_id, :trust_refund_id, :now, :now )" +
                " returning *", (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public ListF<UUID> findInInitStatus(Option<UUID> from, int batchSize) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("status", RefundStatus.INIT.value());
        params.put("batch", batchSize);
        String condition = "";
        if (from.isPresent()) {
            condition = " and id > :from ";
            params.put("from", from.get());
        }

        return jdbcTemplate
                .query("select id from refunds where status = :status " + condition + " order by id limit :batch",
                        (rs, num) -> UUID.fromString(rs.getString("id")), params);
    }

    @Override
    public ListF<Refund> getOrderRefunds(UUID orderId) {
        return jdbcTemplate.query("select * from refunds where order_id = ?",
                (rs, rowNum) -> parseRow(rs), orderId);
    }

    @Override
    public void updateStatus(UUID id, RefundStatus newStatus) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("status", newStatus.value());
        params.put("now", Instant.now());
        params.put("id", id);

        jdbcTemplate.update("update refunds set updated_at = :now, status = :status where id = :id", params);
    }
}
