package ru.yandex.chemodan.app.psbilling.core.dao.users.impl;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.UUID;

import javax.annotation.Nonnull;

import org.intellij.lang.annotations.Language;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.AbstractDaoImpl;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserServiceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.entities.users.OrderStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceBillingStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.users.UserServiceEntity;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.ChildSynchronizableRecordJDBCHelper;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.ParentSynchronizableRecordJdbcHelper;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.SynchronizationStatus;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.resultSet.ResultSetExtras;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

public class UserServiceDaoImpl extends AbstractDaoImpl<UserServiceEntity> implements UserServiceDao {
    private ChildSynchronizableRecordJDBCHelper<UserServiceEntity> childSynchronizableRecordJDBCHelper;
    private ParentSynchronizableRecordJdbcHelper<UserServiceEntity> parentSynchronizableRecordJdbcHelper;

    public UserServiceDaoImpl(JdbcTemplate3 jdbcTemplate) {
        super(jdbcTemplate);
        this.childSynchronizableRecordJDBCHelper = new ChildSynchronizableRecordJDBCHelper<>(
                jdbcTemplate, getTableName(), Option.empty(), this::parseRowTranslated);
        this.parentSynchronizableRecordJdbcHelper = new ParentSynchronizableRecordJdbcHelper<>(
                jdbcTemplate, getTableName(), this::parseRowTranslated, "user_service_features",
                (pAlias, cAlias) -> pAlias + ".id = " + cAlias + ".user_service_id");
    }

    @Override
    public ListF<UserServiceEntity> findEnabledByParentId(UUID id) {
        return jdbcTemplate
                .query("select us.* from user_services us join group_service_members m on m.user_service_id = us.id" +
                                " where m.id = ? and us.target = ?",
                        (rs, num) -> parseRow(rs), id, Target.ENABLED.value());
    }

    @Override
    public ListF<UserServiceEntity> findInInitByParentId(UUID parentId) {
        return jdbcTemplate
                .query("select us.* from user_services us join group_service_members m on m.user_service_id = us.id" +
                                " where m.id = ? and us.status = ?",
                        (rs, num) -> parseRow(rs), parentId, SynchronizationStatus.INIT.value());
    }

    @Override
    public void batchInsert(ListF<InsertData> toInsert, Target target) {
        Instant now = Instant.now();
        jdbcTemplate.batchUpdate(
                "insert into user_services(id, created_at,updated_at,user_product_id,uid,next_check_date, " +
                        "due_date, user_product_price_id, " +
                        "target, target_updated_at, status, status_updated_at, actual_enabled_at, actual_disabled_at," +
                        " " +
                        "first_feature_disabled_at, auto_prolong_enabled, billing_status, last_payment_order_id, " +
                        "package_name)" +
                        " values " +
                        "(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, null, null, null, ?, ?, ?, ?)",
                toInsert.map(m -> new Object[]{
                        m.getId(),
                        now,
                        now,
                        m.getUserProductId(),
                        m.getUid(),
                        m.getNextCheckDate() != null ? m.getNextCheckDate().getOrNull() : null,
                        m.getDueDate() != null ? m.getDueDate().getOrNull() : null,
                        m.getUserProductPriceId() != null ? m.getUserProductPriceId().getOrNull() : null,
                        target.value(),
                        now,
                        SynchronizationStatus.INIT.value(),
                        now,
                        m.getAutoProlongEnabled() == null ? null : m.getAutoProlongEnabled().getOrNull(),
                        m.getBillingStatus() == null ? null :
                                m.getBillingStatus().map(UserServiceBillingStatus::value).getOrNull(),
                        m.getPaidByOrderId() == null ? null : m.getPaidByOrderId().getOrNull(),
                        m.getPackageName() == null ? null : m.getPackageName().getOrNull()
                })
        );
    }

    @Override
    public ParentSynchronizableRecordJdbcHelper<UserServiceEntity> getParentSynchronizableRecordDaoHelper() {
        return parentSynchronizableRecordJdbcHelper;
    }

    @Override
    public ChildSynchronizableRecordJDBCHelper<UserServiceEntity> getChildSynchronizableRecordDaoHelper() {
        return childSynchronizableRecordJDBCHelper;
    }

    @Override
    public ListF<UserServiceEntity> findNotSynchronizedAt(Option<UUID> from,
                                                          int batchSize, Instant syncDate) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("target_not", Target.DISABLED.value());
        params.put("batch", batchSize);
        params.put("types", Cf.x(BillingType.values()).filter(BillingType::isInappProduct).map(BillingType::value));
        params.put("sync_date", syncDate);
        String condition = "";
        if (from.isPresent()) {
            condition = " and us.id > :from ";
            params.put("from", from.get());
        }

        return jdbcTemplate
                .query("select us.* from user_services us " +
                                " join user_products up on up.id = us.user_product_id " +
                                " join orders o on o.id = us.last_payment_order_id" +
                                " where us.target <> :target_not" +
                                " and up.billing_type in ( :types )" +
                                " and (o.inapp_synced_at is null" +
                                " or o.inapp_synced_at < :sync_date)" +
                                " " + condition +
                                " order by us.id limit :batch",
                        (rs, num) -> parseRow(rs), params);
    }

    @Override
    public ListF<UserServiceEntity> findUserServicesForCheck(Option<UUID> from, int batchSize) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("target_not", Target.DISABLED.value());
        params.put("batch", batchSize);
        params.put("now", Instant.now());
        String condition = "";
        if (from.isPresent()) {
            condition = " and id > :from ";
            params.put("from", from.get());
        }

        return jdbcTemplate
                .query("select * from user_services where target <> :target_not and next_check_date < :now " +
                                condition +
                                " order by id limit :batch",
                        (rs, num) -> parseRow(rs), params);
    }

    @Override
    public ListF<UserServiceEntity> find(String uid, Option<UUID> productOwnerId, Option<Target> target,
                                         Option<OrderStatus> orderStatus) {
        MapF<String, Object> params = Cf.hashMap();
        String join = "";
        String condition = "";
        if (productOwnerId.isPresent()) {
            join += " join user_products p on p.id = us.user_product_id ";
            condition += " and p.product_owner_id = :owner_id";
            params.put("owner_id", productOwnerId.get());
        }
        if (target.isPresent()) {
            condition += " and us.target = ( :target ) ";
            params.put("target", target.get().value());
        }
        if (orderStatus.isPresent()) {
            join += " join orders o on o.id = us.last_payment_order_id ";
            condition += " and o.status = :order_status";
            params.put("order_status", orderStatus.get().value());
        }

        params.put("uid", uid);
        @Language("SQL")
        String query = "select us.* from user_services us " + join + " where us.uid = :uid " + condition;
        return jdbcTemplate.query(query, (rs, num) -> parseRow(rs), params);
    }

    @Nonnull
    @Override
    public UserServiceEntity insert(InsertData insertData, Target target) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", now);
        params.put("id", insertData.getId() == null ? UUID.randomUUID() : insertData.getId());
        params.put("user_product_id", insertData.getUserProductId());
        params.put("uid", insertData.getUid());
        params.put("target", target.value());
        params.put("next_check_date",
                insertData.getNextCheckDate() != null ? insertData.getNextCheckDate().getOrNull() : null);
        params.put("due_date",
                insertData.getDueDate() != null ? insertData.getDueDate().getOrNull() : null);
        params.put("user_product_price_id",
                insertData.getUserProductPriceId() != null ? insertData.getUserProductPriceId().getOrNull() : null);
        params.put("status", SynchronizationStatus.INIT.value());
        params.put("auto_prolong_enabled",
                insertData.getAutoProlongEnabled() == null ? null : insertData.getAutoProlongEnabled().getOrNull());
        params.put("billing_status", insertData.getBillingStatus() == null ? null :
                insertData.getBillingStatus().map(UserServiceBillingStatus::value).getOrNull());
        params.put("last_payment_order_id", insertData.getPaidByOrderId() == null ? null :
                insertData.getPaidByOrderId().getOrNull());
        params.put("package_name",
                insertData.getPackageName() == null ? null : insertData.getPackageName().getOrNull());

        return jdbcTemplate.query(
                "insert into user_services(id, created_at,updated_at,user_product_id,uid,next_check_date, " +
                        "due_date, user_product_price_id, " +
                        "target, target_updated_at, status, status_updated_at, actual_enabled_at, actual_disabled_at," +
                        "first_feature_disabled_at, auto_prolong_enabled, billing_status, last_payment_order_id, " +
                        "package_name) " +
                        "values(:id, :now, :now,:user_product_id,:uid,:next_check_date," +
                        " :due_date, :user_product_price_id ," +
                        " :target, :now, :status, :now, null,null, null, :auto_prolong_enabled, :billing_status, " +
                        " :last_payment_order_id, :package_name) " +
                        "RETURNING *",
                (rs, num) -> parseRow(rs), params).first();
    }

    @Override
    public void stopAutoProlong(UUID id, Instant subscriptionUntil) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", now);
        params.put("subscription_until", subscriptionUntil);
        params.put("id", id);

        jdbcTemplate.update("update user_services set updated_at = :now, auto_prolong_enabled = false, " +
                " next_check_date = :subscription_until, due_date = :subscription_until" +
                " where id = :id", params);
    }

    @Override
    public ListF<UUID> findActiveServicesByUserProduct(Option<UUID> fromUserServiceId, UUID userProductId,
                                                       int batchSize) {
        MapF<String, Object> params = Cf.hashMap();
        params.put("product_id", userProductId);
        params.put("batch", batchSize);
        params.put("target_enabled", Target.ENABLED.value());

        String condition = "";
        if (fromUserServiceId.isPresent()) {
            condition = " and id > :id";
            params.put("id", fromUserServiceId.get());
        }

        return jdbcTemplate
                .query("select id from user_services where user_product_id = :product_id and target = :target_enabled "
                                + condition + " order by id limit :batch", (rs, num) -> UUID.fromString(rs.getString(
                                "id")),
                        params);
    }

    @Override
    public void stopAutoProlong(UUID id) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", now);
        params.put("id", id);

        jdbcTemplate.update("update user_services set updated_at = :now, auto_prolong_enabled = false " +
                "where id = :id", params);
    }

    @Override
    public void updateSubscriptionStatus(
            UUID id, UserServiceBillingStatus billingStatus, Instant dueDate, Instant nextCheckDate
    ) {
        Instant now = Instant.now();
        MapF<String, Object> params = Cf.hashMap();
        params.put("now", now);
        params.put("due_date", dueDate);
        params.put("next_check_date", nextCheckDate);
        params.put("id", id);
        params.put("status", billingStatus.value());

        jdbcTemplate.update("update user_services set updated_at = :now," +
                "next_check_date = :next_check_date, due_date = :due_date, " +
                "billing_status = :status where id = :id", params);
    }

    @Override
    public ListF<PassportUid> findActiveUids(ListF<PassportUid> uids) {
        if (uids.isEmpty()) {
            return Cf.list();
        }
        MapF<String, Object> params = Cf.hashMap();
        params.put("uids", uids.map(PassportUid::toString));
        params.put("targetEnabled", Target.ENABLED);

        return jdbcTemplate.query(
                "select uid from user_services where uid in (:uids) and target = :targetEnabled",
                (rs, num) -> PassportUid.cons(Long.parseLong(rs.getString("uid"))),
                params);
    }

    @Override
    public int countServicesWithCheckBillingDateBefore(Duration triggeredBefore) {
        return jdbcTemplate
                .queryForInt("select count(*) from user_services where target = ? and next_check_date is not null " +
                        "and next_check_date < ?", Target.ENABLED.value(), Instant.now().minus(triggeredBefore));
    }

    @Override
    public void updateFirstFeatureDisabledAt(ListF<UUID> ids) {
        if (ids.isEmpty()) {
            return;
        }

        jdbcTemplate.update(
                "update user_services set first_feature_disabled_at = :now " +
                        "where id in ( :ids ) and first_feature_disabled_at is null and target = :target",
                Cf.map(
                        "now", Instant.now(),
                        "ids", ids,
                        "target", Target.DISABLED.value()
                )
        );
    }

    @Override
    public String getTableName() {
        return "user_services";
    }

    public UserServiceEntity parseRow(ResultSet rs) throws SQLException {
        ResultSetExtras extras = new ResultSetExtras(rs);
        Option<Timestamp> actualEnabled = Option.ofNullable(rs.getTimestamp("actual_enabled_at"));
        Option<Timestamp> actualDisabled = Option.ofNullable(rs.getTimestamp("actual_disabled_at"));
        String billingStatus = rs.getString("billing_status");
        String lastPaymentOrderId = rs.getString("last_payment_order_id");
        String packageName = rs.getString("package_name");


        return new UserServiceEntity(
                UUID.fromString(rs.getString("id")),
                new Instant(rs.getTimestamp("created_at")),
                new Instant(rs.getTimestamp("updated_at")),
                Target.R.fromValue(rs.getString("target")),
                new Instant(rs.getTimestamp("target_updated_at")),
                SynchronizationStatus.R.fromValue(rs.getString("status")),
                new Instant(rs.getTimestamp("status_updated_at")),
                actualEnabled.map(Instant::new),
                actualDisabled.map(Instant::new),
                UUID.fromString(rs.getString("user_product_id")),
                rs.getString("uid"),
                Option.ofNullable(rs.getTimestamp("next_check_date")).map(Instant::new),
                Option.ofNullable(rs.getTimestamp("due_date")).map(Instant::new),
                Option.ofNullable(rs.getTimestamp("first_feature_disabled_at")).map(Instant::new),
                Option.ofNullable(rs.getString("user_product_price_id")).map(UUID::fromString),
                extras.getBooleanO("auto_prolong_enabled"),
                StringUtils.isBlank(billingStatus) ? Option.empty() :
                        Option.of(billingStatus).map(UserServiceBillingStatus.R::fromValue),
                StringUtils.isBlank(lastPaymentOrderId) ?
                        Option.empty() : Option.of(UUID.fromString(lastPaymentOrderId)),
                StringUtils.isBlank(packageName) ? Option.empty() : Option.of(packageName)
        );
    }
}
