package ru.yandex.chemodan.app.psbilling.core.directory;

import lombok.AllArgsConstructor;
import org.jetbrains.annotations.NotNull;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice.TooBigGroupException;
import ru.yandex.chemodan.directory.client.DirectoryClient;
import ru.yandex.chemodan.directory.client.DirectoryOrganizationFeaturesResponse;
import ru.yandex.chemodan.directory.client.FetchLimitExceededException;
import ru.yandex.chemodan.directory.client.OrganizationNotFoundException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class DirectoryService {
    private static final Logger logger = LoggerFactory.getLogger(DirectoryService.class);
    private final DirectoryClient directoryClient;
    private final GroupDao groupDao;

    private final DynamicProperty<Integer> directoryPageSize =
            new DynamicProperty<>("ps-billing.directory-client.page-size", 500);
    private final DynamicProperty<Integer> organizationSizeLimit =
            new DynamicProperty<>("ps-billing.directory-client.fetch-organization-size-limit", 100_000);
    private static final int GET_USER_ORGS_PAGE_SIZE = 10;

    // Является ли плательщик админом хотя бы в одной из организаций, где он плательщик?
    public boolean isPayerAdminInOrgs(PassportUid payer) {
        SetF<String> groupsWherePayer = Cf.toSet(
                groupDao.findGroupsByPaymentInfoUid(payer).map(Group::getExternalId));
        if (groupsWherePayer.isEmpty()) {
            return false;
        }
        SetF<String> groupsWhereBelongs = Cf.toSet(
                directoryClient.organizationsWhereUserIsAdmin(
                        payer.toString(), 10));

        return groupsWherePayer.intersects(groupsWhereBelongs);
    }

    @NotNull
    public GroupMembersInfo getGroupMembersInfo(Group group) throws TooBigGroupException {
        try {
            final SetF<String> groupMembers = directoryClient.usersInOrganization(
                    group.getExternalId(), true, directoryPageSize.get(), Option.of(organizationSizeLimit.get()))
                    .map(String::valueOf).unique();
            return new GroupMembersInfo(groupMembers);
        } catch (OrganizationNotFoundException e) {
            logger.warn("Organization {} was empty deleted", group.getExternalId());
            return new GroupMembersInfo(Cf.set(), false);
        } catch (FetchLimitExceededException e) {
            throw new TooBigGroupException(group, e);
        }
    }

    public ListF<String> organizationsWhereUserIsAdmin(PassportUid uid) {
        return directoryClient.organizationsWhereUserIsAdmin(uid.toString(), GET_USER_ORGS_PAGE_SIZE);
    }

    public DirectoryOrganizationFeaturesResponse getOrganizationFeatures(Group group) {
        return directoryClient.getOrganizationFeatures(group.getExternalId());
    }
}
