package ru.yandex.chemodan.app.psbilling.core.entities;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import lombok.Getter;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Period;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.lang.DefaultObject;

@Getter
public class CustomPeriod extends DefaultObject {
    public static final Pattern CUSTOM_PERIOD_PATTERN = Pattern.compile("(\\d+)(\\w+)");

    private final CustomPeriodUnit unit;
    private final int value;

    public CustomPeriod(CustomPeriodUnit unit, int value) {
        this.unit = unit;
        this.value = value;
    }

    public String toTrustPeriod() {
        return (value * unit.getTrustPeriodValue()) + unit.getTrustPeriodUnit();
    }

    public Period toJodaPeriod() {
        return unit.toJodaPeriod().multipliedBy(value);
    }

    public Duration toDurationFrom(Instant instant) {
        return toJodaPeriod().toDurationFrom(instant);
    }

    @Override
    public String toString() {
        return value * unit.getTrustPeriodValue() + unit.getTrustPeriodUnit();
    }

    public static CustomPeriod fromSeconds(int value) {
        return new CustomPeriod(CustomPeriodUnit.ONE_SECOND, value);
    }

    public static CustomPeriod fromMinutes(int value) {
        return new CustomPeriod(CustomPeriodUnit.ONE_MINUTE, value);
    }

    public static CustomPeriod fromHours(int value) {
        return new CustomPeriod(CustomPeriodUnit.ONE_HOUR, value);
    }

    public static CustomPeriod fromDays(int value) {
        return new CustomPeriod(CustomPeriodUnit.ONE_DAY, value);
    }

    public static CustomPeriod fromWeeks(int value) {
        return new CustomPeriod(CustomPeriodUnit.ONE_WEEK, value);
    }

    public static CustomPeriod fromMonths(int value) {
        return new CustomPeriod(CustomPeriodUnit.ONE_MONTH, value);
    }

    public static CustomPeriod fromYears(int value) {
        return new CustomPeriod(CustomPeriodUnit.ONE_YEAR, value);
    }

    public static Option<CustomPeriod> fromResultSet(ResultSet rs, String measurementColumn, String valueColumn) throws SQLException {
        int value = rs.getInt(valueColumn);
        if (rs.wasNull()) {
            return Option.empty();
        }
        Option<String> durationMeasurement = Option.ofNullable(rs.getString(measurementColumn));

        return durationMeasurement.isPresent()
                ? Option.of(new CustomPeriod(CustomPeriodUnit.R.fromValue(durationMeasurement.get()), value))
                : Option.empty();
    }

    public static CustomPeriod fromString(String stringRepresentation) {
        Matcher matcher = CUSTOM_PERIOD_PATTERN.matcher(stringRepresentation);
        if (!matcher.find()) {
            throw new IllegalArgumentException("unable to translate [" + stringRepresentation + "] to custom period");
        }

        return new CustomPeriod(CustomPeriodUnit.fromTrustPeriod(matcher.group(2)), Integer.parseInt(matcher.group(1)));
    }
}
