package ru.yandex.chemodan.app.psbilling.core.entities;

import org.joda.time.Period;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.psbilling.core.texts.PredefinedTankerKey;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.enums.StringEnumResolver;

public enum CustomPeriodUnit implements StringEnum {
    ONE_YEAR("year", "Y", 1, Period.years(1), PredefinedTankerKey.YearPeriod),
    ONE_MONTH("month", "M", 1, Period.months(1), PredefinedTankerKey.MonthPeriod),
    ONE_WEEK("week", "W", 1, Period.weeks(1), PredefinedTankerKey.WeekPeriod),
    ONE_DAY("day", "D", 1, Period.days(1), PredefinedTankerKey.DayPeriod),
    ONE_HOUR("hour", "S", 3600, Period.seconds(1), PredefinedTankerKey.HourPeriod),
    TEN_MINUTES("10minutes", "S", 600,
            Period.seconds(1), PredefinedTankerKey.TenMinutesPeriod), // используется для тестирования, при отдачи
    // фронту мапингся в месячный
    ONE_MINUTE("minute", "S", 60, Period.seconds(1), PredefinedTankerKey.MinutePeriod),
    ONE_SECOND("second", "S", 1, Period.seconds(1), PredefinedTankerKey.SecondPeriod); // для тестирования

    private static final MapF<String, CustomPeriodUnit> trustValues;

    static {
        MapF<String, CustomPeriodUnit> trustValuesInit = Cf.hashMap();
        for (CustomPeriodUnit cpu : CustomPeriodUnit.class.getEnumConstants()) {
            if (cpu.trustPeriodValue == 1) {
                trustValuesInit.put(cpu.getTrustPeriodUnit(), cpu);
            }
        }

        trustValues = trustValuesInit.unmodifiable();
    }

    private final String value;
    private final String trustPeriodUnit;
    private final int trustPeriodValue;
    private final Period trustPeriodUnitAsJoda;
    private final PredefinedTankerKey title;

    CustomPeriodUnit(String value, String trustPeriodUnit, int trustPeriodValue, Period trustPeriodUnitAsJoda,
                     PredefinedTankerKey title) {
        this.value = value;
        this.trustPeriodValue = trustPeriodValue;
        this.trustPeriodUnitAsJoda = trustPeriodUnitAsJoda;
        this.trustPeriodUnit = trustPeriodUnit;
        this.title = title;
    }

    public String toTrustPeriod() {
        return trustPeriodValue + trustPeriodUnit;
    }

    public String getTrustPeriodUnit() {
        return trustPeriodUnit;
    }

    public Period getTrustPeriodUnitAsJoda() {
        return trustPeriodUnitAsJoda;
    }

    public int getTrustPeriodValue() {
        return trustPeriodValue;
    }

    public PredefinedTankerKey getTitle() {
        return title;
    }

    public Period toJodaPeriod() {
        return trustPeriodUnitAsJoda.multipliedBy(trustPeriodValue);
    }

    @Override
    public String value() {
        return value;
    }

    public static final StringEnumResolver<CustomPeriodUnit> R = StringEnumResolver.r(CustomPeriodUnit.class);

    public static CustomPeriodUnit fromTrustPeriod(String trustPeriodUnit) {
        return trustValues.getOrThrow(trustPeriodUnit, "no CustomPeriodUnit for value " + trustPeriodUnit);
    }
}
