package ru.yandex.chemodan.app.psbilling.core.groups;

import java.math.BigDecimal;
import java.util.ConcurrentModificationException;
import java.util.Objects;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServicePriceOverrideDao;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupPaymentType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.PriceOverrideReason;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialUsage;
import ru.yandex.chemodan.app.psbilling.core.mail.EventMailType;
import ru.yandex.chemodan.app.psbilling.core.mail.GroupServiceData;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.Utils;
import ru.yandex.chemodan.app.psbilling.core.products.BucketContentManager;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductQuery;
import ru.yandex.chemodan.app.psbilling.core.products.LimitedTimeProductLineB2b;
import ru.yandex.chemodan.app.psbilling.core.products.TrialDefinition;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductFeature;
import ru.yandex.chemodan.app.psbilling.core.promos.groups.AbstractGroupPromoTemplate;
import ru.yandex.chemodan.app.psbilling.core.promos.v2.GroupPromoUsedService;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.synchronization.groupservice.GroupServicesActualizationService;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.util.BatchFetchingUtils;
import ru.yandex.chemodan.app.psbilling.core.util.LockService;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.MoscowTime;

import static ru.yandex.chemodan.app.psbilling.core.mail.Utils.toDuration;
import static ru.yandex.chemodan.app.psbilling.core.products.UserProductFeatureRegistry.isMpfsSpaceFeature;

@RequiredArgsConstructor
public class GroupServicesManager {
    private static final Logger logger = LoggerFactory.getLogger(GroupServicesManager.class);
    private static final int GROUP_SERVICES_BATCH_SIZE = 1000;
    private static final int GROUP_BATCH_SIZE = 1000;
    private static final String GROUP_PRODUCT_SET_CODE = "mail_pro_b2b";

    private final DynamicProperty<String> b2bBalanceExhaustedLatelyEmailInterval = new DynamicProperty<>(
            "ps-billing-b2b-balance-exhausted-email-interval", CustomPeriod.fromHours(3).toString(),
            Utils::validateCustomPeriod);
    private final DynamicProperty<String> b2bUpsaleEmailInterval = new DynamicProperty<>(
            "ps-billing-b2b-upsale-email-interval", CustomPeriod.fromHours(3).toString(),
            Utils::validateCustomPeriod);

    private final GroupServiceDao groupServiceDao;
    private final GroupServicesActualizationService groupServicesActualizationService;
    private final LockService lockService;
    private final TrialService trialService;
    private final GroupProductManager groupProductManager;
    private final GroupServicePriceOverrideDao groupServicePriceOverrideDao;
    private final BucketContentManager bucketContentManager;
    private final GroupDao groupDao;
    private final TaskScheduler taskScheduler;
    private final GroupPromoUsedService groupPromoUsedService;

    public GroupServiceWrapper wrap(GroupService groupService) {
        return new GroupServiceWrapper(groupService, groupProductManager);
    }

    public ListF<GroupService> find(UUID groupId, Target... targets) {
        return groupServiceDao.find(groupId, targets);
    }

    public ListF<GroupService> find(ListF<UUID> groupIds, Target... targets) {
        return groupServiceDao.find(groupIds, targets);
    }

    public ListF<GroupService> find(UUID groupId, boolean withSubgroups, Target... targets) {
        return groupServiceDao.find(groupId, withSubgroups, targets);
    }


    public MapF<GroupService, GroupProduct> findWithProducts(UUID groupId, boolean withSubgroups, Target... targets) {
        ListF<GroupService> groupServices = groupServiceDao.find(groupId, withSubgroups, targets);
        MapF<GroupService, GroupProduct> result = Cf.linkedHashMap();
        MapF<UUID, GroupProduct> productCache =
                groupProductManager.findByIds(groupServices.map(GroupService::getGroupProductId))
                .toMapMappingToKey(GroupProduct::getId);
        for (GroupService groupService : groupServices) {
            result.put(groupService, productCache.getOrThrow(groupService.getGroupProductId()));
        }
        return result;
    }

    public void disableGroupService(GroupService service) {
        if (service.getTarget() == Target.DISABLED) {
            logger.info("Service {} already disabled", service);
            return;
        }
        if (groupServiceDao.setTargetToDisabled(service.getId(), service.getTarget())) {
            groupServicesActualizationService.scheduleForceGroupActualization(service.getGroupId());
        } else {
            throw new ConcurrentModificationException();
        }
    }

    public GroupService createGroupService(SubscriptionContext context) {
        // lock on group: check group_services and group services creation must be executed in critical section with
        // mutex = group
        return lockService.doWithLockedInTransaction(context.getGroup().getId(), context.getUid(), group -> {
            Option<GroupService> alreadyCreated =
                    findAlreadyCreated(group, context.getProduct(), context.getDeduplicationKey());

            if (alreadyCreated.isPresent()) {
                return alreadyCreated.get();
            }

            disableConflictingServices(group, context.getProduct());

            Option<AbstractGroupPromoTemplate> usedPromo = context.getUsedPromo()
                    .flatMapO(u -> groupPromoUsedService.useGroupPromoForProduct(
                                    context.getUid(),
                                    group,
                                    context.getProduct(),
                                    u.getProductExperiments()
                            )
                    );

            if(usedPromo.isPresent()){
                logger.info("Used promo when create group service for group {}. " +
                                "PromoTemplate code={}",
                        group, usedPromo.get().getCode());
            }

            return createGroupService(
                    group,
                    context.getProduct(),
                    context.getUid(),
                    context.getDeduplicationKey(),
                    !context.isEduGroup(),
                    usedPromo.isPresent(),
                    context.isSendEmail()
            );
        });
    }

    private GroupService createGroupService(
            Group group,
            GroupProduct product,
            PassportUid author,
            Option<String> deduplicationKey,
            boolean checkUserTrial,
            boolean usedPromo,
            boolean sendEmail
    ) {
        Option<Instant> nextBillingDate = Option.of(Instant.now());
        if (product.isFree()) {
            nextBillingDate = Option.empty();
        }

        GroupService created = groupServiceDao.insert(GroupServiceDao.InsertData.builder()
                .groupId(group.getId())
                .productId(product.getId())
                .target(Target.ENABLED)
                .nextBDate(nextBillingDate)
                .deduplicationKey(deduplicationKey.orElse((String) null))
                .skipTransactionsExport(product.isSkipTransactionsExport())
                .hidden(product.isHidden())
                .build()
        );
        if (product.getTrialDefinitionId().isPresent()) {
            TrialDefinition trialDefinition = trialService.findDefinition(product.getTrialDefinitionId().get());
            Option<PassportUid> user = checkUserTrial ? Option.of(author) : Option.empty();
            Option<TrialUsage> usagesO = trialService.findOrCreateTrialUsage(
                    trialDefinition,
                    group,
                    user,
                    usedPromo
            );
            usagesO.forEach(u -> createServicePriceOverrides(created, u, trialDefinition));
        }
        groupServicesActualizationService.scheduleForceGroupActualization(created.getGroupId());

        if (sendEmail) {
            scheduleLocalizedEmailsForPaymentsAwareMembersOfOrganization(created, group, "welcome_paid_360");
        }

        return created;
    }

    public void scheduleEmailForGroupServicesWithEndedTrialInADayMsk(LocalDate day, EventMailType eventMailType) {
        scheduleEmailForGroupServicesWithEndedTrialInInterval(day.toDateTimeAtStartOfDay(MoscowTime.TZ).toInstant(),
                day.plusDays(1).toDateTimeAtStartOfDay(MoscowTime.TZ).toInstant(),
                eventMailType);
    }

    public void scheduleEmailForGroupServicesWithEndedTrialInInterval(Instant after, Instant before,
                                                                      EventMailType eventMailType) {
        ListF<GroupService> services = BatchFetchingUtils.collectBatchedEntities(
                (batchSize, groupServiceId) -> groupServiceDao.findGroupServicesWithNotHiddenTrialOverrideEndDateInBetween(after, before, batchSize, groupServiceId),
                GroupService::getGroupId,
                GROUP_SERVICES_BATCH_SIZE,
                logger
        );
        MapF<UUID, ListF<GroupService>> servicesByGroup = services.groupBy(GroupService::getGroupId);
        ListF<Group> groups = getGroupsForIds(servicesByGroup.keys());
        ListF<Group> immutableGroups = groups.filter(group -> GroupType.ORGANIZATION.equals(group.getType()));
        SetF<UUID> groupIds = immutableGroups.map(Group::getId).unique();
        servicesByGroup = servicesByGroup.filterKeys(groupIds::containsTs);
        servicesByGroup.forEach((groupId, groupServices) ->
                groupServices.forEach(groupService ->
                        taskScheduler.scheduleTransactionalEmailTask(
                                eventMailType,
                                MailContext.builder()
                                        .to(immutableGroups.find(group -> group.getId().equals(groupService.getGroupId()))
                                                .flatMapO(Group::getPaymentInfoWithUidBackwardCompatibility)
                                                .map(BalancePaymentInfo::getPassportUid)
                                                .orElseThrow(IllegalStateException::new))
                                        .groupIds(Cf.list(groupService.getGroupId().toString()))
                                        .groupServices(Cf.list(GroupServiceData.fromGroupService(groupService)))
                                        .userServiceId(Option.empty())
                                        .language(Option.empty())
                                        .build()
                                ,
                                Instant.now().plus(Duration.standardHours(1))))
        );
    }

    public void scheduleBalanceExhaustedEmails(GroupService service, Group group) {
        scheduleLocalizedEmailsForPaymentsAwareMembersOfOrganization(
                service, group, "balance_exhausted"
        );
    }

    public boolean haveActivePostpaidGroupServices(ListF<UUID> groups) {
        ListF<GroupService> postpaidServices = groupServiceDao.findActiveGroupServicesByPaymentType(
                groups, GroupPaymentType.POSTPAID);
        return postpaidServices.filterNot(GroupService::isHidden).isNotEmpty();
    }

    public void sendB2bUpsaleEmails(Instant intervalEnd) {
        InstantInterval interval = new InstantInterval(
                intervalEnd.minus(toDuration(b2bUpsaleEmailInterval)),
                intervalEnd
        );

        ListF<GroupService> services = BatchFetchingUtils.collectBatchedEntities(
                (Integer batchSize, Option<UUID> groupId) ->
                        groupServiceDao.findActiveGroupServicesCreatedInBetween(interval, groupId, batchSize),
                GroupService::getGroupId,
                GROUP_SERVICES_BATCH_SIZE,
                logger
        );
        MapF<UUID, GroupService> serviceByGroup = services.toMap(GroupService::getGroupId, Function.identityF());
        MapF<UUID, Group> groupById = getGroupsForIds(serviceByGroup.keys())
                .toMap(Group::getId, Function.identityF());
        SetF<UUID> groupProductIds = services.map(GroupService::getGroupProductId).unique();
        MapF<UUID, GroupProduct> groupProductById = groupProductManager.findByIds(groupProductIds)
                .toMap(GroupProduct::getId, Function.identityF());
        MapF<UUID, ListF<GroupProduct>> groupProductsByLineId = Cf.hashMap();

        serviceByGroup.forEach((groupId, groupService) -> {
            Group group = groupById.getTs(groupId);
            PassportUid ownerUid = group.getOwnerUid();
            GroupProduct boughtProduct = groupProductById.getTs(groupService.getGroupProductId());
            GroupProductQuery query = new GroupProductQuery(
                    GROUP_PRODUCT_SET_CODE,
                    Option.of(ownerUid),
                    Option.of(group),
                    Cf.list(),
                    Option.empty()
            );
            Option<LimitedTimeProductLineB2b> limitedLineB2bO = groupProductManager.findProductLineWithPromo(query);
            if (limitedLineB2bO.isEmpty()) {
                return;
            }
            UUID lineId = limitedLineB2bO.get().getProductLine().getId();
            if (!groupProductsByLineId.containsKeyTs(lineId)) {
                groupProductsByLineId.put(lineId, groupProductManager.getProductsByLineId(lineId));
            }
            ListF<GroupProduct> products = groupProductsByLineId.getTs(lineId);
            Option<GroupProduct> nextProduct = getNextProduct(boughtProduct, products);

            nextProduct.ifPresent(p -> taskScheduler.scheduleB2bUpsaleEmailTask(ownerUid, boughtProduct, p));
        });
    }

    private Option<GroupProduct> getNextProduct(GroupProduct boughtProduct, ListF<GroupProduct> products) {
        Option<UserProductFeature> boughtSpaceFeature = boughtProduct.getFeatures()
                .find(feature -> isMpfsSpaceFeature(feature.getCode()));
        if (boughtSpaceFeature.isEmpty()) {
            return Option.empty();
        }
        BigDecimal currentProductSpaceAmount = boughtSpaceFeature.get().getAmount();
        return products.filter(p -> boughtProduct.getPricePerUserInMonth().compareTo(p.getPricePerUserInMonth()) <= 0)
                .filter(p -> p.getFeatures()
                        .find(feature -> isMpfsSpaceFeature(feature.getCode()))
                        .map(UserProductFeature::getAmount)
                        .filter(amount -> amount.compareTo(currentProductSpaceAmount) > 0)
                        .isPresent()
                )
                .sortedBy(GroupProduct::getPricePerUserInMonth)
                .firstO();
    }

    public void sendPaymentReminderEmails(LocalDate today) {
        for (Tuple2<String, LocalDate> params : Tuple2List.fromPairs(
                "payment_reminder_3_days", today.plusDays(3),
                "payment_reminder_7_days", today.plusDays(7),
                "payment_reminder_10_days", today.plusDays(10)
        )) {
            scheduleEmailsForGroupsWithActivePrepaidServicesAndClientBalanceVoidIn(params.get2(), params.get1());
        }
    }

    private void scheduleEmailsForGroupsWithActivePrepaidServicesAndClientBalanceVoidIn(
            LocalDate date, String emailTemplateKey
    ) {
        DateTimeZone tz = MoscowTime.TZ;
        InstantInterval interval = new InstantInterval(
                date.toDateTimeAtStartOfDay(tz),
                date.plusDays(1).toDateTimeAtStartOfDay(tz)
        );

        BatchFetchingUtils.fetchAndProcessBatchedEntities(
                (Integer batchSize, Option<UUID> groupId) ->
                        groupDao.findGroupsWithActivePrepaidService(
                                interval, groupId, batchSize),
                Group::getId,
                group -> scheduleEmailsForGroupWithActivePrepaidServicesAndClientBalanceVoidIn(group, date,
                        emailTemplateKey),
                GROUP_SERVICES_BATCH_SIZE,
                logger
        );
    }

    private void scheduleEmailsForGroupWithActivePrepaidServicesAndClientBalanceVoidIn(Group group, LocalDate date,
                                                                                       String emailTemplateKey) {
        if (group.getPaymentInfo().exists(BalancePaymentInfo::isB2bAutoBillingEnabled)) {
            logger.debug("Skipping {} for group {} with auto billing", emailTemplateKey, group.getId());
            return;
        }

        MailContext context = MailContext.builder()
                .to(group.getOwnerUid())
                .balanceVoidDate(Option.of(date))
                .build();

        taskScheduler.schedulePaymentsAwareMembersEmailTask(
                group, emailTemplateKey, context, Option.of(Duration.standardDays(25))
        );
    }

    public void scheduleB2bBalanceExhaustedLatelyEmails(String emailKey, Instant intervalStart) {
        InstantInterval interval = new InstantInterval(
                intervalStart.minus(toDuration(b2bBalanceExhaustedLatelyEmailInterval)), intervalStart);
        BatchFetchingUtils.fetchAndProcessBatchedEntities(
                (Integer batchSize, Option<UUID> groupId) ->
                        groupDao.findGroupsWithDisabledPrepaidGsAndNoActiveGs(
                                interval, groupId, batchSize),
                Group::getId,
                group -> scheduleB2bBalanceExhaustedEmailWithClientBalanceVoidIn(emailKey, group),
                GROUP_SERVICES_BATCH_SIZE,
                logger
        );
    }

    private void scheduleB2bBalanceExhaustedEmailWithClientBalanceVoidIn(String emailKey, Group group) {
        PassportUid ownerUid = group.getOwnerUid();
        taskScheduler.scheduleB2bBalanceExhaustedLatelyEmail(ownerUid, emailKey);
        Option<PassportUid> payerUid = group.getPaymentInfo()
                .map(BalancePaymentInfo::getPassportUid);
        if (payerUid.isPresent() && !payerUid.isSome(ownerUid)) {
            taskScheduler.scheduleB2bBalanceExhaustedLatelyEmail(payerUid.get(), emailKey);
        }
    }

    private ListF<Group> getGroupsForIds(ListF<UUID> ids) {
        ListF<Group> groups = Cf.arrayList();
        for (int i = 0; i * GROUP_BATCH_SIZE < ids.size(); i++) {
            int startIndex = i * GROUP_BATCH_SIZE;
            groups.addAll(groupDao.findGroups(ids.subList(startIndex, Math.min(startIndex + GROUP_BATCH_SIZE,
                    ids.size()))));
        }
        return groups.unmodifiable();
    }

    private boolean scheduleLocalizedEmailsForPaymentsAwareMembersOfOrganization(
            GroupService service, Group group, String emailKey
    ) {
        if (GroupType.ORGANIZATION.equals(group.getType())
                && !service.isSkipTransactionsExport()
                && !service.isHidden()
        ) {
            MailContext context = MailContext.builder()
                    .to(group.getOwnerUid())
                    .groupIds(Cf.list(group.getId().toString()))
                    .groupServices(Cf.list(GroupServiceData.fromGroupService(service)))
                    .build();

            taskScheduler.schedulePaymentsAwareMembersEmailTask(group, emailKey, context, Option.empty());
            return true;
        }
        return false;
    }

    private Option<GroupService> findAlreadyCreated(Group group, GroupProduct product,
                                                    Option<String> deduplicationKey) {
        ListF<GroupService> groupServices = groupServiceDao.find(group.getId(), product.getId());

        // check for retry (with help of deduplication key)
        if (deduplicationKey.isPresent() && !StringUtils.isBlank(deduplicationKey.get())) {
            String key = deduplicationKey.get();
            Option<GroupService> alreadyCreatedO = groupServices
                    .filter(groupService -> groupService.getDeduplicationKey().isPresent())
                    .find(groupService -> Objects.equals(groupService.getDeduplicationKey().get(), key));

            if (alreadyCreatedO.isPresent()) {
                //found retry - group service with the same deduplication key
                GroupService alreadyCreated = alreadyCreatedO.get();
                if (alreadyCreated.getTarget() != Target.ENABLED) {
                    throw new A3ExceptionWithStatus("deduplication_conflict", HttpStatus.SC_409_CONFLICT);
                } else {
                    logger.info("product {} already purchased for organization {} with deduplication key {}",
                            product, group, deduplicationKey);
                    return alreadyCreatedO;
                }
            }
        }

        //check for singleton service
        if (product.isSingleton()) {
            Option<GroupService> alreadyCreatedO =
                    groupServices.find(groupService -> groupService.getTarget() == Target.ENABLED);
            if (alreadyCreatedO.isPresent()) {
                return alreadyCreatedO;
            }
        }

        return Option.empty();
    }


    private void disableConflictingServices(Group group, GroupProduct product) {
        //find buckets with our product
        ListF<SetF<String>> buckets = bucketContentManager.getBuckets().filter(s -> s.containsTs(product.getCode()));
        if (buckets.isEmpty()) {
            return;
        }
        SetF<String> conflictingCodes = buckets.flatMap(Function.identityF()).unique();

        //find all active services of our group
        ListF<GroupService> groupServices = groupServiceDao.find(group.getId(), Target.ENABLED);
        if (groupServices.isEmpty()) {
            return;
        }

        //find products of active services
        MapF<UUID, GroupProduct> products =
                groupProductManager.findByIds(groupServices.map(GroupService::getGroupProductId).unique())
                        .toMapMappingToKey(GroupProduct::getId);

        // disabling  all services in our bucket
        groupServices
                .filter(gs -> conflictingCodes.containsTs(products.getOrThrow(gs.getGroupProductId()).getCode()))
                .forEach(this::disableGroupService);
    }

    private void createServicePriceOverrides(GroupService created, TrialUsage u, TrialDefinition trialDefinition) {
        groupServicePriceOverrideDao.insert(GroupServicePriceOverrideDao.InsertData.builder()
                .pricePerUserInMonth(trialDefinition.getPrice())
                .endDate(Option.of(u.getEndDate()))
                .groupServiceId(created.getId())
                .hidden(trialDefinition.isHidden())
                .reason(PriceOverrideReason.TRIAL)
                .startDate(created.getCreatedAt())
                .trialUsageId(Option.of(u.getId()))
                .build());
    }
}
