package ru.yandex.chemodan.app.psbilling.core.groups;

import java.util.List;
import java.util.UUID;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalTime;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.ClientBalanceCalculator;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupPartnerDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServicePriceOverrideDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductOwnerDao;
import ru.yandex.chemodan.app.psbilling.core.entities.AbstractEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServicePriceOverride;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductOwner;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductManager;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.util.LockService;
import ru.yandex.chemodan.directory.client.DirectoryClient;
import ru.yandex.chemodan.directory.client.DirectoryOrganizationByIdResponse;
import ru.yandex.chemodan.directory.client.DirectoryOrganizationFeaturesResponse;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.chemodan.util.exception.NotFoundException;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@RequiredArgsConstructor
public class GroupsManager {
    private static final Logger logger = LoggerFactory.getLogger(GroupsManager.class);
    private final GroupDao groupDao;
    private final GroupProductManager groupProductManager;
    private final ProductOwnerDao productOwnerDao;
    private final GroupServicePriceOverrideDao groupServicePriceOverrideDao;
    private final GroupServicesManager groupServicesManager;
    private final LockService lockService;
    private final Duration gracePeriod;
    private final DirectoryClient directoryClient;
    private final ClientBalanceCalculator clientBalanceCalculator;
    private final GroupPartnerDao groupPartnerDao;

    public ListF<Group> findGroupsByPayer(PassportUid uid) {
        return groupDao.findGroupsByPaymentInfoUid(uid);
    }

    public Group findById(UUID groupId) {
        return groupDao.findById(groupId);
    }

    @NotNull
    public Option<Group> findGroup(PassportUid uid, GroupType type, String groupExternalId) {
        Option<Group> groupO = findGroupTrusted(type, groupExternalId);

        //TODO https://st.yandex-team.ru/CHEMODAN-69854
//        groupO.ifPresent(group -> {
//            if (!group.getOwnerUid().equalsTs(uid)) {
//                throw new AccessForbiddenException(type.value() + " owned by other uid than '" + uid.getUid() + "'");
//            }
//        });

        return groupO;
    }

    @NotNull
    public Option<Group> findGroupTrusted(GroupType type, String groupExternalId) {
        return groupDao.findGroup(type, groupExternalId);
    }

    public SetF<String> findSubscribedGroupProductCodes(GroupType type, CollectionF<String> groupExternalIds) {
        return groupDao.findGroupProductCodesWithActiveServicesByExternalGroupIds(type, groupExternalIds);
    }

    @NotNull
    public Group findGroupOrThrow(PassportUid uid, GroupType type, String groupExternalId) {
        return findGroup(uid, type, groupExternalId)
                .orElseThrow(() -> getGroupNotFoundException(type, groupExternalId));
    }

    @NotNull
    public Group findGroupTrustedOrThrow(GroupType type, String groupExternalId) {
        return findGroupTrusted(type, groupExternalId)
                .orElseThrow(() -> getGroupNotFoundException(type, groupExternalId));
    }

    private NotFoundException getGroupNotFoundException(GroupType type, String groupExternalId) {
        return new NotFoundException("unknown " + type.value() + " '" + groupExternalId + "'");
    }

    public Instant calculateGroupNextBillingDate(Group group) {
        if (group.getType() == GroupType.ORGANIZATION) {
            // POSTPAID
            return nextBillingDateForOrganization(group);
        } else {
            throw new IllegalStateException();
        }
    }

    public Option<Instant> calculatePaymentPeriodEndDate(Group group) {
        if (group.getType() == GroupType.ORGANIZATION) {
            // POSTPAID
            Instant now = Instant.now();
            Instant lastDayOfPreviousMonth =
                    now.toDateTime().toLocalDate().minusMonths(1).dayOfMonth().withMaximumValue()
                            .toDateTime(LocalTime.MIDNIGHT, DateTimeZone.getDefault()).toInstant();

            return group.getStatus() != ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupStatus.ACTIVE ?
                    Option.of(lastDayOfPreviousMonth.plus(group.getGracePeriod().orElse(Duration.ZERO))) :
                    Option.empty();
        } else {
            throw new IllegalStateException();
        }
    }

    public Instant findFirstPaidDay(GroupService service, ListF<GroupServicePriceOverride> overrides) {
        //сервис совсем не биллим
        if (service.isSkipTransactionsExport()) {
            return Instant.now();
        }

        //ищем первую дату, после которой мы начнем начислять открутки
        overrides = overrides.sortedBy(GroupServicePriceOverride::getStartDate);
        Option<Instant> candidate = Option.empty();
        for (GroupServicePriceOverride override : overrides) {
            if (override.getEndDate().isPresent() && override.getEndDate().get().isBeforeNow()) {
                continue;
            }

            if (override.isHidden()) {
                continue;
            }
            if (!override.getStartDate().isAfterNow()) {
                //нашли оверрайд действующий на сейчас
                if (override.isFree()) {
                    candidate = override.getEndDate(); // TODO если оверрайд бессрочный, мы вернем текущую дату в итоге
                    continue;
                } else {
                    return Instant.now();
                }
            }

            //оверрайд, который будет в будущем
            if (override.isFree()) {
                if (candidate.isPresent()) {
                    //нашли бесплатный  интервал, и до него уже был такой же бесплатный. смотрим, если они встык, или
                    // нет.
                    if (candidate.get().equals(override.getStartDate())) {
                        candidate = override.getEndDate();
                        continue;
                    } else {
                        return candidate.get();
                    }
                } else {
                    //кандидата нет, и текущий бесплатный в будущем. поэтому прямо сейчас -открутки начисляем
                    return Instant.now();
                }
            } else {
                //нашли платный оверрайд, который в будущем
                return candidate.orElseGet(Instant::now);
            }
        }

        return candidate.orElse(Instant.now());
    }

    private Instant nextBillingDateForOrganization(Group group) {
        ListF<GroupService> services =
                groupServicesManager.find(group.getId(), Target.ENABLED).filterNot(GroupService::isHidden);
        MapF<UUID, ListF<GroupServicePriceOverride>> priceOverrides =
                groupServicePriceOverrideDao.findByGroupServices(services.map(GroupService::getId));


        if (services.isEmpty()) {
            return getDay2OfNextMonthMidnight();
        }

        //смотрим если есть оверрайды цен, и когда согласно этим оверрайдам надо начинать билить
        return getDay2OfNextMonthMidnightAfterMoment(findFirstPaidDay(services, priceOverrides));
    }

    private Instant findFirstPaidDay(ListF<GroupService> services,
                                     MapF<UUID, ListF<GroupServicePriceOverride>> priceOverrides) {
        Option<Instant> startOfBilling = Option.empty();
        for (GroupService service : services) {
            Instant firstPaidDay = findFirstPaidDay(service, priceOverrides.getOrElse(service.getId(), Cf.list()));
            if (!startOfBilling.isPresent() || startOfBilling.get().isAfter(firstPaidDay)) {
                startOfBilling = Option.of(firstPaidDay);
            }
        }

        return startOfBilling.orElseGet(Instant::now);
    }

    private static Instant getDay2OfNextMonthMidnightAfterMoment(Instant moment) {
        return moment.toDateTime().toLocalDate().plusMonths(1).withDayOfMonth(2)
                .toDateTime(LocalTime.MIDNIGHT, DateTimeZone.getDefault()).toInstant();
    }

    private static Instant getDay2OfNextMonthMidnight() {
        return getDay2OfNextMonthMidnightAfterMoment(Instant.now());
    }

    @Transactional
    public Group createOrUpdateOrganization(PassportUid authorUid, String organizationId,
                                            boolean isEduGroup, ProductOwner productOwner,
                                            BalancePaymentInfo balancePaymentInfo, Option<String> clid) {
        Group group = createOrUpdateOrganization(authorUid, organizationId, balancePaymentInfo, clid);
        initGroup(group, isEduGroup, authorUid, productOwner);
        return group;
    }

    private Group createOrUpdateOrganization(PassportUid authorUid, String organizationId,
                                             BalancePaymentInfo balancePaymentInfo, Option<String> clid) {
        Option<Group> groupO = groupDao.findGroup(GroupType.ORGANIZATION, organizationId);
        if (groupO.isPresent()) {
            return updatePaymentInfo(balancePaymentInfo, groupO);
        } else {
            Group group = createOrganization(authorUid, organizationId, balancePaymentInfo, clid);
            clientBalanceCalculator.updateClientBalance(balancePaymentInfo.getClientId());
            return group;
        }
    }

    private Group updatePaymentInfo(BalancePaymentInfo balancePaymentInfo, Option<Group> groupO) {
        Group group = groupO.get();
        validatePaymentInfoUpdate(group.getPaymentInfo(), balancePaymentInfo);
        boolean updateClientBalance;
        if (group.getPaymentInfo().isEmpty()) {
            balancePaymentInfo.withB2bAutoBillingEnabled(
                    isAutoBillingEnabledForClient(balancePaymentInfo.getClientId()));
            updateClientBalance = true;
        } else {
            balancePaymentInfo.withB2bAutoBillingEnabled(
                    group.getPaymentInfo().get().isB2bAutoBillingEnabled());
            updateClientBalance = false;
        }
        logger.info("Updating payment info for organization {}: {}", group.getId(), balancePaymentInfo);
        group = groupDao.updatePaymentInfo(group.getId(), balancePaymentInfo);
        if (updateClientBalance) {
            clientBalanceCalculator.updateClientBalance(balancePaymentInfo.getClientId());
        }
        return group;
    }

    private void validatePaymentInfoUpdate(Option<BalancePaymentInfo> existingO, BalancePaymentInfo newPaymentInfo) {
        if (existingO.isEmpty()) {
            return;
        }
        BalancePaymentInfo existing = existingO.get();
        if (!existing.getClientId().equals(newPaymentInfo.getClientId())
                || !existing.getPassportUid().equalsTs(newPaymentInfo.getPassportUid())) {
            logger.error("Updating payment info prohibited, old: {}, new: {}",
                    existing, newPaymentInfo);
            throw new A3ExceptionWithStatus("payment info update prohibited", HttpStatus.SC_400_BAD_REQUEST);
        }
    }

    @NotNull
    private Group createOrganization(PassportUid authorUid, String organizationId,
                                     BalancePaymentInfo balancePaymentInfo,
                                     Option<String> clid) {
        balancePaymentInfo.withB2bAutoBillingEnabled(
                isAutoBillingEnabledForClient(balancePaymentInfo.getClientId())
        );

        Group group = groupDao.insert(GroupDao.InsertData.builder()
                .ownerUid(authorUid).type(GroupType.ORGANIZATION).externalId(organizationId)
                .gracePeriod(gracePeriod)
                .paymentInfo(balancePaymentInfo)
                .clid(clid)
                .build());
        getAndSavePartnerId(group);
        logger.info("Created organization {} with payment info {}", group.getId(), balancePaymentInfo);
        return group;
    }


    public Group getGroupOrCreateWithoutPayment(PassportUid uid, GroupType groupType, String groupExternalId,
                                                Option<String> clid) {
        Option<Group> groupO = groupDao.findGroup(groupType, groupExternalId);
        if (groupO.isPresent()) {
            return groupO.get();
        }

        Group group = groupDao.insert(GroupDao.InsertData.builder()
                .ownerUid(uid).type(groupType).externalId(groupExternalId)
                .gracePeriod(gracePeriod).clid(clid).build());
        getAndSavePartnerId(group);
        logger.info("Created organization {} without payment info", group.getId());
        return group;
    }

    public void initGroup(Group group, boolean isEduGroup, PassportUid authorUid, ProductOwner productOwner) {
        lockService.doWithLockedInTransaction(group.getId(), authorUid, g -> {
            groupDao.insertAgreement(group.getId(), productOwner.getId());
            if (productOwner.getDefaultGroupProductId().isPresent()) {
                GroupProduct defaultProduct =
                        groupProductManager.findById(productOwner.getDefaultGroupProductId().get());
                logger.info("creating default group product {} for group {}", defaultProduct, group);

                Validate.isTrue(defaultProduct.isSingleton());
                groupServicesManager
                        .createGroupService(new SubscriptionContext(defaultProduct, g, authorUid, Option.empty(),
                                isEduGroup));
            }
            return null;
        });
    }

    public void removePaymentData(Group group) {
        logger.info("clearing payment data for {}", group);
        groupDao.removePaymentInfo(group.getId());
    }

    @Transactional
    public void acceptAgreementForGroup(
            PassportUid uid, GroupType groupType, String groupExternalId, boolean isEduGroup, String productOwnerCode,
            Option<String> clid) {
        Group group = getGroupOrCreateWithoutPayment(uid, groupType, groupExternalId, clid);
        initGroup(group, isEduGroup, uid, productOwnerDao.findByCode(productOwnerCode)
                .orElseThrow(() -> new NotFoundException("productOwner not found")));
    }

    public ListF<Group> filterGroupsByAcceptedAgreement(
            GroupType groupType, ListF<String> groupExternalIds, String productOwnerCode) {
        return productOwnerDao.findByCode(productOwnerCode).flatMap(productOwner ->
                groupDao.filterGroupsByAcceptedAgreement(groupType, groupExternalIds, productOwner.getId()));
    }

    public GroupFeatures checkOrgFeatures(String organizationId) {
        DirectoryOrganizationFeaturesResponse organizationFeatures =
                directoryClient.getOrganizationFeatures(organizationId);
        DirectoryOrganizationByIdResponse organization = directoryClient.getOrganizationById(organizationId);
        if (organizationFeatures.isDisablePsbillingProcessingActive() ||
                organization.getOrganizationType().equals("portal")) {
            throw new A3ExceptionWithStatus("paid_product_not_available",
                    "Paid product not available for this organization", HttpStatus.SC_400_BAD_REQUEST);
        }
        return new GroupFeatures(organizationFeatures.isEdu360Active());
    }

    public boolean hasActivePostpaidGroupServices(PassportUid uid) {
        ListF<UUID> payerGroups = findGroupsByPayer(uid).map(AbstractEntity::getId);
        return groupServicesManager.haveActivePostpaidGroupServices(payerGroups);
    }

    public boolean isAutoBillingEnabledForClient(Long clientId) {
        ListF<Group> groups = groupDao.findGroupsByPaymentInfoClient(clientId);
        Option<Group> groupWithAutopayEnabled = groups.filter(g -> g.getPaymentInfo()
                .map(BalancePaymentInfo::isB2bAutoBillingEnabled).orElse(false)).firstO();
        return groupWithAutopayEnabled.isPresent();
    }

    public Option<BalancePaymentInfo> getPaymentInfo(PassportUid payerUid) {
        ListF<Group> groups = findGroupsByPayer(payerUid);
        // костыль, пока не сделаем уникальный payment_info в отдельной таблице
        SetF<BalancePaymentInfo> paymentInfos = groups.map(x -> x.getPaymentInfo().get()).unique();
        if (paymentInfos.size() > 1) {
            logger.warn("found several different payment infos by one uid {}", payerUid.toString());
        }
        return paymentInfos.iterator().nextO();
    }

    public boolean hasPaymentInfoForOrganization(String groupExternalId) {
        return groupDao.findGroup(GroupType.ORGANIZATION, groupExternalId).flatMap(Group::getPaymentInfo).isNotEmpty();
    }

    public boolean isPartnerClient(UUID groupId) {
        return groupPartnerDao.getPartnerId(groupId).isPresent();
    }

    private void getAndSavePartnerId(Group group) {
        Option<String> partnerId = directoryClient.getOrganizationPartnerId(group.getExternalId());
        if (partnerId.isPresent()) {
            groupPartnerDao.insertPartnerInfoIfNotExists(group.getId(), partnerId.get());
        }
    }

    @Transactional
    public List<Group> createSubgroupIfNotExists(Group parentGroup, GroupType subgroupType, List<String> subgroupExternalIds) {
        logger.info("Creating subgroups {} of type {} for parent group {}",
                subgroupExternalIds, subgroupType, parentGroup);
        if (parentGroup.getParentGroupId().isPresent())
            throw new IllegalArgumentException("Multiple group inheritance is prohibited");
        ListF<Group> result = Cf.arrayList();
        for (String subgroupExternalId : subgroupExternalIds) {
            Option<Group> existed = groupDao.findGroup(subgroupType, subgroupExternalId, parentGroup.getId());
            if (existed.isPresent()) {
                result.add(existed.get());
                continue;
            }
            Group group = groupDao.insert(GroupDao.InsertData.builder()
                    .ownerUid(parentGroup.getOwnerUid())
                    .type(subgroupType)
                    .externalId(subgroupExternalId)
                    .paymentInfo(parentGroup.getPaymentInfo().orElse((BalancePaymentInfo) null))
                    .clid(parentGroup.getClid())
                    .gracePeriod(Duration.ZERO)
                    .parentGroupId(Option.of(parentGroup.getId()))
                    .build());
            logger.info("Created subgroup {}", group.getId());
            result.add(group);
        }
        return result;
    }

    @Getter
    @AllArgsConstructor
    public class GroupFeatures {
        private boolean isEdu;
    }
}
