package ru.yandex.chemodan.app.psbilling.core.groups;

import lombok.AllArgsConstructor;
import lombok.RequiredArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.FeatureDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProduct;
import ru.yandex.chemodan.app.psbilling.core.products.GroupProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductFeatureRegistry;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.passport.tvmauth.exception.PermissionDenied;

@RequiredArgsConstructor
public class OptionalFeaturesManager {

    private final GroupServicesManager groupServicesManager;
    private final GroupProductManager groupProductManager;
    private final GroupServiceDao groupServiceDao;
    private final FeatureDao featureDao;

    @AllArgsConstructor
    static class ActivationUnit {
        final String REQUIRED_CODE;
        final String FEATURE_CODE;
        final String PRODUCT_CODE;
    }

    public static final ActivationUnit LETTERS_ARCHIVE_ACTIVATION =
            new ActivationUnit(UserProductFeatureRegistry.LETTER_ARCHIVE_AVAILABLE_FEATURE_CODE,
                    UserProductFeatureRegistry.LETTER_ARCHIVE_ENABLED_FEATURE_CODE, "letters_archive");

    private final MapF<String, ActivationUnit> featureCodeToActivation = Cf.map(
            LETTERS_ARCHIVE_ACTIVATION.FEATURE_CODE, LETTERS_ARCHIVE_ACTIVATION
    );

    public boolean enable(Group group, PassportUid uid, String featureCode) {
        return toggle(group, uid, featureCode, true);
    }

    public boolean disable(Group group, PassportUid uid, String featureCode) {
        return toggle(group, uid, featureCode, false);
    }

    /**
     @return true if state changed
     */
    private boolean toggle(Group group, PassportUid uid, String featureCode, boolean enabledTarget) {
        if (!featureCodeToActivation.containsKeyTs(featureCode)) {
            throw new IllegalStateException("Feature " + featureCode + " is not supported");
        }
        ActivationUnit activationUnit = featureCodeToActivation.getTs(featureCode);
        String requiredCode = activationUnit.REQUIRED_CODE;
        if (!featureDao.isEnabledForGroup(group.getId(), requiredCode)) {
            throw new PermissionDenied("Feature " + requiredCode + " is not available for this group!");
        }


        final String featureActivationCode = activationUnit.FEATURE_CODE;
        boolean enabled = featureDao.isEnabledForGroup(
                group.getId(),
                featureActivationCode
        );
        if (enabled == enabledTarget) {
            return false;
        }
        GroupProduct activationProduct =
                groupProductManager.findProduct(activationUnit.PRODUCT_CODE);

        if (enabledTarget) {
            groupServicesManager.createGroupService(
                    // is edu false cause it doesn't matter now. No need to call Directory for no reason
                    new SubscriptionContext(activationProduct, group, uid, Option.empty(), false)
            );
        } else {
            groupServicesManager.disableGroupService(
                    groupServiceDao.find(group.getId(), activationProduct.getId()).first()
            );
        }
        return true;
    }
}
