package ru.yandex.chemodan.app.psbilling.core.groups;

import java.util.UUID;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.TrialDefinitionDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.TrialUsageDao;
import ru.yandex.chemodan.app.psbilling.core.entities.AbstractEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialDefinitionEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialUsage;
import ru.yandex.chemodan.app.psbilling.core.products.TrialDefinition;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class TrialService {
    private static final Logger logger = LoggerFactory.getLogger(TrialService.class);
    private final TrialDefinitionDao trialDefinitionDao;
    private final TrialUsageDao trialUsageDao;

    public TrialDefinitionEntity insert(TrialDefinitionDao.InsertData insertData) {
        return trialDefinitionDao.insert(insertData);
    }

    public TrialUsage insert(TrialUsageDao.InsertData insertData) {
        return trialUsageDao.insert(insertData);
    }

    public TrialDefinition findDefinition(UUID trialDefinitionId) {
        return new TrialDefinition(trialDefinitionDao.findById(trialDefinitionId));
    }

    public ListF<TrialDefinition> findDefinitions(ListF<UUID> ids) {
        return trialDefinitionDao.findByIds(ids).map(TrialDefinition::new);
    }

    private ListF<TrialUsage> findUsages(String usageComparisonKey, Option<UUID> groupIdO, Option<PassportUid> uidO) {
        return trialUsageDao.findTrialUsages(groupIdO, uidO, usageComparisonKey);
    }

    public ListF<TrialUsage> findUsageForUser(String usageComparisonKey, PassportUid uid) {
        return trialUsageDao.findTrialUsages(Option.empty(), Option.of(uid), usageComparisonKey);
    }

    public TrialAvailability isTrialAvailable(
            TrialDefinition trialDefinition,
            Option<Group> groupO,
            Option<PassportUid> author,
            boolean allowMultipleUsage
    ) {
        if (author.isEmpty() && groupO.isEmpty()) {
            return new TrialAvailability(true, Option.empty());
        }

        ListF<TrialUsage> trialUsages;

        if (trialDefinition.getSingleUsageComparisonKey().isEmpty()) {
            if (groupO.isPresent()) {
                Group group = groupO.get();
                trialUsages = findUsagesByTrialDefinition(trialDefinition.getId(), group.getId());
            } else {
                trialUsages = Cf.list();
            }
        } else {
            String singleKey = trialDefinition.getSingleUsageComparisonKey().get();
            // got all usages by group or by user
            trialUsages = findUsages(singleKey, groupO.map(AbstractEntity::getId), author);
        }

        if (groupO.isPresent()) {
            Group group = groupO.get();
            return isTrialAvailable(trialDefinition, group, trialUsages, allowMultipleUsage);
        }
        // пользователь может активировать триал только один раз, а если группа не передано,
        // и использование уже есть на какой-то группе, то еще раз его не активировать
        return new TrialAvailability(trialUsages.isEmpty(), Option.empty());
    }

    private ListF<TrialUsage> findUsagesByTrialDefinition(UUID trialDefinitionId, UUID groupId) {
        return trialUsageDao.findByTrialDefinitionAndGroup(trialDefinitionId, groupId);
    }

    private TrialAvailability isTrialAvailable(
            TrialDefinition trialDefinition,
            Group group,
            ListF<TrialUsage> trialUsages,
            boolean allowMultipleUsage
    ) {
        // ищем использования триала группой
        ListF<TrialUsage> groupTrialUsages = trialUsages
                .filter(u -> u.getGroupId().map(id -> id.equals(group.getId())).orElse(false))
                .sortedByDesc(AbstractEntity::getCreatedAt);

        Option<TrialUsage> trialUsageO;

        if (trialDefinition.getSingleUsageComparisonKey().isEmpty()) {
            trialUsageO = groupTrialUsages.firstO();
        } else {
            trialUsageO = groupTrialUsages.singleO();
        }

        if (trialUsageO.isPresent()) {
            //использование группы есть - проверяем его, вдруг его можно заново активировать для этой же группы
            //если дата окончания уже в прошлом, но он недоступен
            Instant trialEndDate = trialDefinition.calculateEndDate();
            if (trialEndDate.isBeforeNow()) {
                logger.info("trial end date in the past, so trial unavailable. trial: {}, endDate: {}",
                        trialDefinition, trialEndDate);
                return new TrialAvailability(false, Option.empty());
            }

            TrialUsage trialUsage = trialUsageO.get();
            if (trialUsage.getEndDate().isBeforeNow()) {
                boolean multipleUsage = trialDefinition.getSingleUsageComparisonKey().isEmpty() && allowMultipleUsage;

                logger.info("found trial usage end date in the past, trial unavailable. Is multiple usage {}", multipleUsage);
                return new TrialAvailability(multipleUsage, multipleUsage ? Option.empty() : trialUsageO);
            }

            return new TrialAvailability(true, trialUsageO);
        }
        // у нас найдены использования, но они сделаны другими пользователями, по другим группам.
        // в этом случае триал недоступен - так как нельзя использовать триал одновременно в разных группах
        return new TrialAvailability(trialUsages.isEmpty(), Option.empty());
    }

    public Option<TrialUsage> findOrCreateTrialUsage(TrialDefinition trialDefinition, PassportUid author) {
        return findOrCreateTrialUsage(trialDefinition, Option.empty(), Option.of(author), false);
    }

    public Option<TrialUsage> findOrCreateTrialUsage(TrialDefinition trialDefinition,
                                                     Group group,
                                                     Option<PassportUid> author,
                                                     boolean allowMultipleUsage
    ) {
        return findOrCreateTrialUsage(trialDefinition, Option.of(group), author, allowMultipleUsage);
    }

    /**
     * Если у триала есть single_usage_comparison_key, то используем его для повторного использования триала.
     * Если single_usage_comparison_key отсутствует, то ищем не использованные по groupProductId.
     * Если есть используем повторно. Если таковы отсутствуют, то создаем новый.
     *
     * @param trialDefinition
     * @param group
     * @param author
     * @return
     */
    private Option<TrialUsage> findOrCreateTrialUsage(
            TrialDefinition trialDefinition,
            Option<Group> group,
            Option<PassportUid> author,
            boolean allowMultipleUsage
    ) {
        TrialAvailability available = isTrialAvailable(trialDefinition, group, author, allowMultipleUsage);

        if (!available.isAvailable()) {
            return Option.empty();
        }

        if (available.getUsage().isPresent()) {
            //какое-то ДОСТУПНОЕ использование уже есть, а если оно доступно, то значит было на нашей группу
            return available.getUsage();
        } else {
            return Option.of(insert(TrialUsageDao.InsertData.builder()
                    .activatedByUid(author)
                    .groupId(group.map(Group::getId))
                    .trialDefinitionId(trialDefinition.getId())
                    .endDate(trialDefinition.calculateEndDate())
                    .build())
            );
        }
    }

    @Data
    public static class TrialAvailability {
        private final boolean available;
        private final Option<TrialUsage> usage;
    }
}
