package ru.yandex.chemodan.app.psbilling.core.mail;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Builder;
import lombok.Data;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.StringEntity;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderContext;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderTemplateDefinition;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class SenderClient {

    private static final Logger logger = LoggerFactory.getLogger(SenderClient.class);

    private static final String SEND_EMAIL_URL_TEMPLATE = "/api/0/%s/transactional/%s/send"; // TODO not transactional

    private static final String TO_EMAIL_FIELD = "to_email";

    private final HttpClient httpClient;

    private final ObjectMapper objectMapper;

    private final String senderBaseUrl;

    private final Option<String> blackboxEnv;

    public SenderClient(HttpClient httpClient, ObjectMapper objectMapper, String senderBaseUrl,
                        Option<String> blackboxEnv) {
        this.httpClient = httpClient;
        this.objectMapper = objectMapper;
        this.senderBaseUrl = getUrlWithScheme(senderBaseUrl);
        this.blackboxEnv = blackboxEnv;
    }

    public boolean sendEmail(SenderContext context) {
        HttpUriRequest request = buildRequest(context.getSenderTemplateDefinition(), context, Option.empty());
        SenderResponseData senderResponseData = execute(request);
        while (senderResponseData.isShouldDoRetry()) {
            senderResponseData = execute(
                    buildRequest(context.getSenderTemplateDefinition(), context, Option.of(senderResponseData)));
        }
        return senderResponseData.wasSent;
    }

    private SenderResponseData execute(HttpUriRequest request) {
        try {
            return httpClient.execute(request, this::processSenderResponse);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private SenderResponseData processSenderResponse(HttpResponse response) throws IOException {
        int statusCode = response.getStatusLine().getStatusCode();
        String content = response.getEntity() == null
                ? "<no content>"
                : IOUtils.toString(response.getEntity().getContent(), StandardCharsets.UTF_8);
        if (HttpStatus.is2xx(statusCode)) {
            logger.info("The sender request has been processed successfully with code={} response={}", statusCode,
                    content);
            return SenderResponseData.noRetry(true);
        }
        if (statusCode == HttpStatus.SC_410_GONE) {
            logger.warn("The email has not been sent because it is disabled {}", content);
            return SenderResponseData.noRetry(false);
        }
        if (statusCode == HttpStatus.SC_400_BAD_REQUEST && isInvalidEmail(content)) {
            logger.warn("Retrying without email response={}", content);
            return SenderResponseData.retryWithoutEmail();
        }
        logger.warn("Sender sent an error statusCode={} response={}", statusCode, content);
        if (HttpStatus.is4xx(statusCode)) {
            logger.warn("No retry");
            return SenderResponseData.noRetry(false);
        }
        throw new RuntimeException(String.format("Bad sender response status=%s", statusCode));
    }

    private boolean isInvalidEmail(String responseBody) {
        MapF<Object, Object> error = getError(responseBody);
        if (!error.containsKeyTs(TO_EMAIL_FIELD)) {
            return false;
        }
        Object description = error.getTs(TO_EMAIL_FIELD);
        if (!(description instanceof List)) {
            return false;
        }
        List<Object> descriptionList = (List) description;
        if (descriptionList.isEmpty()) {
            return false;
        }
        return "Invalid value".equals(descriptionList.get(0));
    }

    private MapF<Object, Object> getError(String responseBody) {
        MapF<String, Object> response = Cf.map();
        try {
            response = Cf.toHashMap(objectMapper.<Map<String, Object>>readValue(responseBody,
                    new TypeReference<HashMap<String, Object>>() {
                    }));
        } catch (IOException e) {
            logger.warn("Cannot parse the sender response body {}", responseBody);
        }
        return Cf.x(response.getO("result")
                .filter(Map.class::isInstance)
                .map(Map.class::cast)
                .map(result -> result.getOrDefault("error", Cf.map()))
                .filter(Map.class::isInstance)
                .map(Map.class::cast)
                .getOrElse(Collections::emptyMap));
    }

    private HttpUriRequest buildRequest(SenderTemplateDefinition templateDefinition, SenderContext context,
                                        Option<SenderResponseData> senderResponseData) {
        Option<PassportUid> toUser = Option.ofNullable(context.getTo());
        Option<String> toEmail = senderResponseData.exists(SenderResponseData::isWithoutEmail)
                ? Option.empty() : Option.ofNullable(context.getEmail());

        Check.isTrue(toUser.isPresent() || toEmail.isPresent(), "Missing recipient");

        UriBuilder builder = UriBuilder.cons(senderBaseUrl)
                .appendPath(String.format(SEND_EMAIL_URL_TEMPLATE,
                                templateDefinition.getAccount(),
                                templateDefinition.getCampaign()));
        toUser.forEach(uid ->
                builder.addParam("to_yandex_puid", uid));
        blackboxEnv.forEach(env ->
                builder.addParam("blackbox_env", env));
        toEmail.forEach(email ->
                builder.addParam(TO_EMAIL_FIELD, email));

        HttpPost request = new HttpPost(builder.build());
        try {
            request.setEntity(new StringEntity(objectMapper.writeValueAsString(new SenderRequestBody(context)),
                    StandardCharsets.UTF_8));
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
        request.addHeader("X-Sender-I-Am-Sender", "true");
        request.addHeader(HttpHeaders.CONTENT_TYPE, "application/json");
        request.addHeader("X-Sender-Prefer-Account", templateDefinition.getAccount());
        return request;
    }

    private String getUrlWithScheme(String url) {
        if (url.startsWith("http://") || url.startsWith("https://")) {
            return url;
        }
        return "https://" + url;
    }

    @Data
    @Builder
    private static class SenderResponseData {

        public static SenderResponseData noRetry(boolean wasSend) {
            return SenderResponseData.builder()
                    .shouldDoRetry(false)
                    .withoutEmail(false)
                    .wasSent(wasSend).build();
        }

        public static SenderResponseData retryWithoutEmail() {
            return SenderResponseData.builder()
                    .shouldDoRetry(true)
                    .withoutEmail(true)
                    .wasSent(false).build();
        }

        private final boolean shouldDoRetry;
        private final boolean withoutEmail;
        private final boolean wasSent;
    }
}
