package ru.yandex.chemodan.app.psbilling.core.mail.dataproviders;

import java.io.IOException;
import java.math.BigDecimal;
import java.net.URI;
import java.util.Base64;
import java.util.UUID;

import lombok.Data;
import org.apache.commons.io.IOUtils;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.balance.PaymentData;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.GroupBillingService;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.GroupBillingStatus;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.InvoiceFile;
import ru.yandex.chemodan.app.psbilling.core.billing.groups.Money;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupStatus;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.mail.Attachment;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderContext;
import ru.yandex.chemodan.app.psbilling.core.mail.localization.DateResolver;
import ru.yandex.chemodan.app.psbilling.core.mail.localization.TermsResolver;
import ru.yandex.chemodan.balanceclient.model.response.GetClientContractsResponseItem;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


public abstract class AbstractCheckDebtsSenderDataProvider extends AbstractSenderDataProvider {
    private static final Logger logger = LoggerFactory.getLogger(AbstractCheckDebtsSenderDataProvider.class);
    private static final String YOUR_ORGANIZATION = "yourOrganization";

    private final GroupDao groupDao;
    private final DateResolver dateResolver;
    private final GroupBillingService groupBillingService;
    private final BalanceService balanceService;
    private final TermsResolver termsResolver;
    private final FeatureFlags featureFlags;

    public AbstractCheckDebtsSenderDataProvider(GroupDao groupDao, DateResolver dateResolver,
                                                Blackbox2 blackbox2, GroupBillingService groupBillingService,
                                                BalanceService balanceService,
                                                TermsResolver termsResolver, GroupsManager groupsManager,
                                                FeatureFlags featureFlags) {
        super(blackbox2, groupsManager);
        this.groupDao = groupDao;
        this.dateResolver = dateResolver;
        this.groupBillingService = groupBillingService;
        this.balanceService = balanceService;
        this.termsResolver = termsResolver;
        this.featureFlags = featureFlags;
    }

    @Override
    public Option<SenderContext> buildSenderContext(MailContext mailContext) {
        PassportUid to = mailContext.getTo();
        Language language = mailContext.getLanguage().getOrThrow(IllegalStateException::new);
        ListF<Group> groups = groupDao.findGroups(mailContext.getGroupIds().map(UUID::fromString));
        ListF<Group> groupsWithDifferentBalanceClientIds = getGroupsWithDifferentClientIds(groups);
        ListF<GroupDebtData> groupDebtDataList = groupsWithDifferentBalanceClientIds.map(this::getGroupDebtData)
                .filter(Option::isPresent)
                .map(Option::get);
        if (groupDebtDataList.isEmpty()) {
            logger.info("No debt for {}", mailContext);
            return Option.empty();
        }
        ListF<Attachment> attachments = groupDebtDataList.flatMap(GroupDebtData::getInvoices);
        MapF<String, ListF<BigDecimal>> debtAmountToCurrency = groupDebtDataList
                .map(GroupDebtData::getDebtAmount)
                .groupByMapValues(Money::getCurrencyCode, Money::getAmount);
        if (debtAmountToCurrency.size() > 1) {
            throw new IllegalStateException(String.format("User %s has contracts with different currencies %s", to,
                    debtAmountToCurrency.keySet()));
        }
        String currency = debtAmountToCurrency.entries().first().get1();
        BigDecimal amount = debtAmountToCurrency.values()
                .flatMap(Function.identityF())
                .stream()
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        MapF<Long, PaymentData> paymentData = balanceService.findPaymentData(groupsWithDifferentBalanceClientIds
                .find(group -> group.getPaymentInfo().isPresent())
                .getOrThrow(IllegalStateException::new)
                .getPaymentInfo().get().getClientId());
        PaymentData balancePaymentData = paymentData.entries().first().get2();
        Instant paymentDeadlineInstant = groupDebtDataList.map(x -> x.payDeadline).min();
        String paymentDeadline = dateResolver.getLocalizedMskDate(paymentDeadlineInstant, language);
        String publicDisplayName = getPublicNameForUid(to);

        String yourOrganizationTerm = termsResolver.getManyFormOfTerm(YOUR_ORGANIZATION, language, groups.size());
        return Option.of(SenderContext.builder()
                .args(Cf.<String, Object>map(
                        "public_display_name", publicDisplayName,
                        "payment_deadline", paymentDeadline,
                        "sum", getMoneySum(amount),
                        "currency", processCurrency(getCurrencyByCode(currency.toUpperCase()))
                ).plus1("your_organization_s", yourOrganizationTerm))
                .attachments(attachments)
                .to(to)
                .email(getEmail(balancePaymentData))
                .build()
        );
    }

    protected String processCurrency(String currency) {
        return currency.replaceAll("\\.+$", "");
    }

    protected abstract GroupStatus getValidBillingStatus();

    private Option<GroupDebtData> getGroupDebtData(Group group) {
        GroupBillingStatus billingStatus = groupBillingService.actualGroupBillingStatus(group);
        GroupStatus expectedBillingStatus = getValidBillingStatus();
        if (!billingStatus.getStatus().equals(expectedBillingStatus)) {
            logger.info("The group {} has not valid billing status: expected {} actual {}",
                    group, expectedBillingStatus, billingStatus);
            return Option.empty();
        }
        MapF<Long, Money> contractRequiredPayments = billingStatus.getContractRequiredPayments();
        MapF<String, ListF<Money>> moneyToCurrency =
                contractRequiredPayments.entries().map(Tuple2::get2).groupBy(Money::getCurrencyCode);
        if (moneyToCurrency.size() > 1) {
            throw new IllegalStateException(String.format("Group %s has contracts with different currencies %s",
                    group, moneyToCurrency.keySet()));
        }
        String currency = moneyToCurrency.entries().first().get1();
        BigDecimal amount = moneyToCurrency.getTs(currency).map(Money::getAmount).stream()
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        BalancePaymentInfo paymentInfo = group.getPaymentInfoWithUidBackwardCompatibility()
                .getOrThrow("there must be payment info for email creation");
        ListF<Attachment> invoices = createAttachments(paymentInfo, contractRequiredPayments);
        return Option.of(new GroupDebtData(new Money(amount, currency), invoices,
                billingStatus.getFirstDebtPaymentDeadline().get()));
    }

    private ListF<Attachment> createAttachments(BalancePaymentInfo paymentInfo,
                                                MapF<Long, Money> contractRequiredPayments) {
        MapF<Long, GetClientContractsResponseItem> contracts =
                balanceService.getClientContracts(paymentInfo.getClientId())
                        .toMapMappingToKey(GetClientContractsResponseItem::getId);
        ListF<InvoiceFile> invoices = contractRequiredPayments
                .entries()
                .map(entry -> groupBillingService.createInvoice(paymentInfo, contracts.getOrThrow(entry.get1()),
                        entry.get2()));
        return invoices.map(this::createAttachmentForInvoice);
    }

    private Attachment createAttachmentForInvoice(InvoiceFile invoice) {
        try {
            byte[] fileContent = IOUtils.toByteArray(URI.create(invoice.getUrl()));
            return Attachment.builder()
                    .fileName(invoice.getFileName())
                    .data(Base64.getEncoder().encodeToString(fileContent))
                    .mimeType("application/pdf")
                    .build();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private ListF<Group> getGroupsWithDifferentClientIds(ListF<Group> groups) {
        return groups
                .filter(group -> group.getPaymentInfoWithUidBackwardCompatibility().map(BalancePaymentInfo::getClientId).isPresent())
                .groupBy(group -> group.getPaymentInfoWithUidBackwardCompatibility().map(BalancePaymentInfo::getClientId).get())
                .mapValues(groupList -> groupList.first())
                .values()
                .toList();
    }

    @Data
    private static class GroupDebtData {
        private final Money debtAmount;
        private final ListF<Attachment> invoices;
        private final Instant payDeadline;
    }
}
