package ru.yandex.chemodan.app.psbilling.core.mail.dataproviders;

import java.util.UUID;
import java.util.function.Supplier;

import com.google.common.base.Preconditions;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductOwnerDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.texts.TankerKeyDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupType;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductOwner;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.texts.TankerTranslationEntity;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.Utils;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderTemplateDefinition;
import ru.yandex.chemodan.balanceclient.model.response.FindClientResponseItem;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.utils.Language;

public abstract class AbstractSenderByProductOwnerDataProvider extends AbstractSenderDataProvider {

    private final ProductOwnerDao productOwnerDao;

    private final UserProductDao userProductDao;

    private final BalanceService balanceService;

    private final GroupDao groupDao;

    private final GroupProductDao groupProductDao;

    private final TankerKeyDao tankerKeyDao;

    private final MapF<String, Supplier<MapF<Language, DynamicProperty<String>>>> templatesByLanguageForProductOwners =
            Cf.map(MAIL_PRODUCT_OWNER_CODE, this::getTemplatesByLanguageForMail,
                    "yandex_disk", this::getTemplatesByLanguageForDisk);

    public AbstractSenderByProductOwnerDataProvider(Blackbox2 blackbox2, GroupsManager groupsManager,
            ProductOwnerDao productOwnerDao, UserProductDao userProductDao, BalanceService balanceService,
            GroupDao groupDao, GroupProductDao groupProductDao, TankerKeyDao tankerKeyDao)
    {
        super(blackbox2, groupsManager);
        this.productOwnerDao = productOwnerDao;
        this.userProductDao = userProductDao;
        this.balanceService = balanceService;
        this.groupDao = groupDao;
        this.groupProductDao = groupProductDao;
        this.tankerKeyDao = tankerKeyDao;
    }

    @Override
    public Option<SenderTemplateDefinition> selectMailTemplateDefinition(MailContext mailContext) {
        Preconditions.checkArgument(mailContext.getGroupServices().size() == 1,
                "Empty list of group services or more than one %s");
        ProductOwner productOwner = getProductOwner(UUID.fromString(mailContext.getGroupServices().first().getGroupServiceId()));
        MapF<Language, DynamicProperty<String>> templatesByLanguage = templatesByLanguageForProductOwners
                .getO(productOwner.getCode()).map(Supplier::get).getOrElse(Cf::map);
        return templatesByLanguage.getO(mailContext.getLanguage().getOrThrow(IllegalStateException::new))
                .map(DynamicProperty::get)
                .map(Utils::parseTemplateDefinition);
    }

    protected UserProductDao getUserProductDao() {
        return this.userProductDao;
    }

    protected abstract MapF<Language, DynamicProperty<String>> getTemplatesByLanguageForDisk();

    protected abstract MapF<Language, DynamicProperty<String>> getTemplatesByLanguageForMail();

    protected DynamicProperty<String> getRuSenderTemplate() {
        return null;
    }

    protected void checkMailContextPreconditionsForOneGroupService(MailContext mailContext) {
        Preconditions.checkArgument(mailContext.getGroupIds().size() == 1,
                "Empty list of groups or more than one %s", mailContext.getGroupIds());
        Preconditions.checkArgument(mailContext.getGroupServices().size() == 1,
                "Empty list of group services or more than one %s", mailContext.getGroupServices());
    }

    protected FindClientResponseItem getBalanceClientDataForOrganization(Group organization) {
        return balanceService.findClientData(
                organization.getPaymentInfoWithUidBackwardCompatibility().map(BalancePaymentInfo::getPassportUid)
                        .getOrThrow(String.format("No payment data for organization %s", organization.getId()))
        ).getOrThrow(String.format("No balance clients for group %s", organization.getId()));
    }

    protected Group getOrganizationById(UUID id) {
        Group organization = groupDao.findById(id);
        Preconditions.checkArgument(GroupType.ORGANIZATION.equals(organization.getType()),
                "Group %s is not an organization", organization);
        return organization;
    }

    protected String getTariffName(UUID groupServiceId, Language language) {
        GroupProductEntity groupProductEntity = groupProductDao.findByGroupServiceId(groupServiceId);
        ListF<TankerTranslationEntity> tankerTranslations =
                tankerKeyDao.findTranslations(groupProductEntity.getTitleTankerKeyId()
                        .getOrThrow("No tanker data for user product"));
        return tankerTranslations.filter(tankerTranslation -> tankerTranslation.getLocale().equals(language.value()))
                .first().getTranslationText();
    }

    protected GroupProductDao getGroupProductDao() {
        return this.groupProductDao;
    }

    private ProductOwner getProductOwner(UUID groupServiceId) {
        UserProductEntity userProductEntity = userProductDao.findByGroupServiceId(groupServiceId);
        return productOwnerDao.findById(userProductEntity.getProductOwnerId());
    }
}
