package ru.yandex.chemodan.app.psbilling.core.mail.dataproviders;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.function.Supplier;

import org.apache.commons.lang3.StringUtils;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.balance.PaymentData;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.Utils;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderTemplateDefinition;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductFeature;
import ru.yandex.chemodan.balanceclient.model.response.FindClientResponseItem;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDbFields;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.net.LocalhostUtils;

public abstract class AbstractSenderDataProvider implements SenderDataProvider {

    public static final String MAIL_PRODUCT_OWNER_CODE = "yandex_mail";

    private static final DecimalFormat moneyFormat = new DecimalFormat("0.00");

    private static final DecimalFormat nonFractionMoneyFormat = new DecimalFormat("0");

    private final MapF<Language, Supplier<DynamicProperty<String>>> templatesByLanguage =
            Cf.map(Language.RUSSIAN, this::getRuSenderTemplate);

    private final MapF<String, String> currenciesRepresentation = Cf.map(
            "RUB", "руб.",
            "USD", "$");
    private final Blackbox2 blackbox2;

    private final GroupsManager groupsManager;

    public AbstractSenderDataProvider(Blackbox2 blackbox2, GroupsManager groupsManager) {
        this.blackbox2 = blackbox2;
        this.groupsManager = groupsManager;
    }

    @Override
    public Option<SenderTemplateDefinition> selectMailTemplateDefinition(MailContext mailContext) {
        return templatesByLanguage.getO(mailContext.getLanguage().getOrThrow(IllegalStateException::new))
                .map(Supplier::get)
                .map(DynamicProperty::get)
                .map(Utils::parseTemplateDefinition);
    }

    public Option<Language> getLanguage(PassportUid uid) {
        return Option.of(Language.RUSSIAN);
    }

    protected String getPublicNameForUid(PassportUid uid) {
        BlackboxCorrectResponse userInfoResponse = blackbox2.query().userInfo(
                LocalhostUtils.localAddress(),
                Option.of(uid),
                Option.empty(), Option.empty(),
                Cf.list(BlackboxDbFields.FIRSTNAME),
                Cf.list(),
                Option.empty(),
                Option.empty(),
                true,
                Option.empty(),
                false,
                true
        );
        return userInfoResponse
                .getDisplayName()
                .map(blackboxDisplayName -> blackboxDisplayName.getPublicName().getOrElse(blackboxDisplayName.getName()))
                .getOrElse(userInfoResponse.getDbFields().getO(BlackboxDbFields.FIRSTNAME).getOrElse(""));
    }

    protected Instant getNextBillingDateForOrganization(Group group) {
        return groupsManager.calculateGroupNextBillingDate(group);
    }


    protected String getMoneySum(BigDecimal sum) {
        if (sum.remainder(BigDecimal.ONE).equals(BigDecimal.ZERO)) {
            return nonFractionMoneyFormat.format(sum);
        }
        return moneyFormat.format(sum);
    }

    protected ListF<String> getFeatureNames(UserProduct product, Language language, boolean appendValue) {
        return product.getFeatures()
                .filter(UserProductFeature::isEnabled)
                .map(feature -> feature.getEmailRepresentation(language, appendValue))
                .filter(Option::isPresent)
                .map(Option::get);
    }

    protected String getCurrencyByCode(String code) {
        return currenciesRepresentation.getO(code).getOrElse(code);
    }

    protected GroupsManager getGroupsManager() {
        return groupsManager;
    }

    protected String getEmail(PaymentData paymentData) {
        return getEmail(paymentData.getEmail());
    }

    protected String getEmail(FindClientResponseItem findClientResponseItem) {
        return getEmail(findClientResponseItem.getEmail());
    }

    protected abstract DynamicProperty<String> getRuSenderTemplate();

    private String getEmail(String email) {
        return Option.ofNullable(email)
                .filter(StringUtils::isNotBlank)
                .map(Utils::parseFirstEmailFromBalance)
                .filter(StringUtils::isNotBlank).getOrNull();
    }
}
