package ru.yandex.chemodan.app.psbilling.core.mail.dataproviders;

import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductOwnerDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.texts.TankerKeyDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.mail.EventMailType;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderContext;
import ru.yandex.chemodan.app.psbilling.core.mail.localization.DateResolver;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.balanceclient.model.response.FindClientResponseItem;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.chemodan.app.psbilling.core.mail.Utils.DELIMITER;

public class B2bJustPaidOrEndTrialSenderDataProvider extends AbstractSenderByProductOwnerDataProvider {

    private final static Logger logger = LoggerFactory.getLogger(B2bJustPaidOrEndTrialSenderDataProvider.class);

    private final DateResolver dateResolver;
    private final UserProductManager userProductManager;
    private final GroupServiceDao groupServiceDao;
    private final FeatureFlags featureFlags;

    private final DynamicProperty<String> mailRuSenderTemplateDefinition =
            new DynamicProperty<>("ps-billing-transactional-emails-mail-just-paid-ru-template-definition",
                    "disk" + DELIMITER + "WY06TSN3-TUK");

    private final DynamicProperty<String> diskRuSenderTemplateDefinition =
            new DynamicProperty<>("ps-billing-transactional-emails-disk-just-paid-ru-template-definition",
                    "disk" + DELIMITER + "8QC2SSN3-ILN");

    private MapF<Language, DynamicProperty<String>> mailTemplatesByLanguage =
            Cf.map(Language.RUSSIAN, mailRuSenderTemplateDefinition);

    private MapF<Language, DynamicProperty<String>> diskTemplatesByLanguage =
            Cf.map(Language.RUSSIAN, diskRuSenderTemplateDefinition);

    public B2bJustPaidOrEndTrialSenderDataProvider(Blackbox2 blackbox2,
                                                   GroupDao groupDao, DateResolver dateResolver,
                                                   GroupProductDao groupProductDao, UserProductDao userProductDao,
                                                   TankerKeyDao tankerKeyDao, BalanceService balanceService,
                                                   ProductOwnerDao productOwnerDao,
                                                   UserProductManager userProductManager,
                                                   GroupsManager groupsManager, GroupServiceDao groupServiceDao,
                                                   FeatureFlags featureFlags)
    {
        super(blackbox2, groupsManager, productOwnerDao, userProductDao, balanceService, groupDao,
                groupProductDao, tankerKeyDao);
        this.dateResolver = dateResolver;
        this.userProductManager = userProductManager;
        this.groupServiceDao = groupServiceDao;
        this.featureFlags = featureFlags;
    }

    @Override
    public Option<SenderContext> buildSenderContext(MailContext mailContext) {
        PassportUid to = mailContext.getTo();
        Language language = mailContext.getLanguage().getOrThrow(IllegalStateException::new);

        checkMailContextPreconditionsForOneGroupService(mailContext);
        UUID groupId = UUID.fromString(mailContext.getGroupIds().first());
        UUID groupServiceId = UUID.fromString(mailContext.getGroupServices().first().getGroupServiceId());
        GroupService groupService = groupServiceDao.findById(groupServiceId);
        if (!groupService.getTarget().equals(Target.ENABLED)) {
            logger.info("The service {} is not enabled", groupService);
            return Option.empty();
        }

        String publicDisplayName = getPublicNameForUid(to);

        String tariffName = getTariffName(groupServiceId, language);

        Group organization = getOrganizationById(groupId);
        FindClientResponseItem balancePaymentData = getBalanceClientDataForOrganization(organization);
        String organizationName = balancePaymentData.getName();

        UserProduct userProduct =
                userProductManager.findById(getUserProductDao().findByGroupServiceId(groupServiceId).getId());
        ListF<String> features = getFeatureNames(userProduct, language, false);

        Instant nextBillingDate = getNextBillingDateForOrganization(organization);
        String billSendDate = dateResolver.getLocalizedMskDate(nextBillingDate, language);

        MapF<String, Object> args = Cf.map(
                "public_display_name", publicDisplayName,
                "tariff_name", tariffName,
                "org_name", organizationName);

        return Option.of(SenderContext.builder()
                .args(args.plus1("features", features)
                        .plus1("bill_send_date", billSendDate))
                .attachments(Cf.list())
                .to(to)
                .email(getEmail(balancePaymentData))
                .build()
        );
    }

    @Override
    public boolean isAcceptableFor(EventMailType eventMailType, MailContext context) {
        return EventMailType.B2B_JUST_PAID_OR_END_TRIAL.equals(eventMailType);
    }

    @Override
    protected MapF<Language, DynamicProperty<String>> getTemplatesByLanguageForDisk() {
        return diskTemplatesByLanguage;
    }

    @Override
    protected MapF<Language, DynamicProperty<String>> getTemplatesByLanguageForMail() {
        return mailTemplatesByLanguage;
    }
}
