package ru.yandex.chemodan.app.psbilling.core.mail.dataproviders;

import java.util.UUID;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.mail.EventMailType;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderContext;
import ru.yandex.chemodan.app.psbilling.core.mail.localization.DateResolver;
import ru.yandex.chemodan.app.psbilling.core.mail.localization.TermsResolver;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.chemodan.app.psbilling.core.mail.Utils.DELIMITER;

public class JustPaidSenderDataProvider extends AbstractPochtaProSenderDataProvider {

    private static final Logger logger = LoggerFactory.getLogger(JustPaidSenderDataProvider.class);

    private final UserProductManager userProductManager;

    private final TermsResolver termsResolver;

    private final DateResolver dateResolver;

    private final DynamicProperty<String> senderRuTemplateDefinition =
            new DynamicProperty<>("ps-billing-transactional-emails-pochta-just-paid-ru-template-definition",
                    "Yandex.Mail" + DELIMITER + "8YPXBSN3-WHQ");

    public JustPaidSenderDataProvider(Blackbox2 blackbox2, GroupsManager groupsManager, DateResolver dateResolver,
                                      UserServiceManager userServiceManager, UserProductManager userProductManager,
                                      TermsResolver termsResolver) {
        super(blackbox2, groupsManager, userServiceManager);
        this.userProductManager = userProductManager;
        this.termsResolver = termsResolver;
        this.dateResolver = dateResolver;
    }

    @Override
    protected DynamicProperty<String> getRuSenderTemplate() {
        return senderRuTemplateDefinition;
    }

    @Override
    public Option<SenderContext> buildSenderContext(MailContext mailContext) {
        checkMailContextPreconditions(mailContext);

        PassportUid to = mailContext.getTo();
        Language language = mailContext.getLanguage().getOrThrow(IllegalStateException::new);
        UserService userService = mailContext.getUserServiceId()
                .map(UUID::fromString)
                .map(getUserServiceManager()::findById)
                .get();
        if (!userService.getPrice().isPresent()) {
            logger.info("Empty price for {}", userService);
            return Option.empty();
        }

        String publicDisplayName = getPublicNameForUid(to);

        String tariffName = getTariffName(userService, language);

        UserProductPrice productPrice = userService.getPrice().getOrThrow(IllegalStateException::new);
        ListF<String> features = getFeatureNames(productPrice.getPeriod().getUserProduct(), language, true);

        String period = termsResolver.getTermForCustomPeriodPeriod(productPrice.getPeriod().getPeriod(), language);
        String cost = getMoneySum(productPrice.getPrice());
        String currency = productPrice.getCurrencyCode();

        String nextPurchaseDate = userService.getNextCheckDate()
                .map(checkDate -> dateResolver.getLocalizedMskDate(checkDate, language))
                .getOrThrow(IllegalStateException::new);

        return Option.of(SenderContext.builder()
                .args(Cf.<String, Object>map(
                                "public_display_name", publicDisplayName,
                                "features", features,
                                "tariff_name", tariffName,
                                "next_purchase_date", nextPurchaseDate
                        ).plus1("period", period)
                        .plus1("cost", cost)
                        .plus1("currency", getCurrencyByCode(currency.toUpperCase())))
                .attachments(Cf.list())
                .to(to)
                .build()
        );
    }

    @Override
    public boolean isAcceptableFor(EventMailType eventMailType, MailContext context) {
        return EventMailType.JUST_PAID.equals(eventMailType);
    }
}
