package ru.yandex.chemodan.app.psbilling.core.mail.dataproviders;

import java.util.Arrays;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.chemodan.app.psbilling.core.dao.mail.EmailTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.entities.mail.EmailTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.mail.LocalizedEmailTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.Utils;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderContext;
import ru.yandex.chemodan.app.psbilling.core.mail.keydataproviders.EmailKeyDataProvider;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDbFields;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.net.LocalhostUtils;

public class LocalizedEmailSenderDataProvider {
    private static final Logger logger = LoggerFactory.getLogger(LocalizedEmailSenderDataProvider.class);

    private final EmailTemplateDao emailTemplateDao;
    private final Blackbox2 blackbox2;
    private final MapF<String, EmailKeyDataProvider> keyProvidersMap;

    private final DynamicProperty<String> emailsLanguageMappings = new DynamicProperty<>(
            "ps-billing-emails-language-mappings",
            "ru,uk,be->*,ru; *->*,en",
            this::validateLanguageMappings
    );

    public LocalizedEmailSenderDataProvider(
            EmailTemplateDao emailTemplateDao,
            ListF<EmailKeyDataProvider> emailKeyDataProviders,
            Blackbox2 blackbox2
    ) {
        this.emailTemplateDao = emailTemplateDao;
        this.blackbox2 = blackbox2;
        keyProvidersMap = createKeyProvidersMap(emailKeyDataProviders);
    }

    public Option<SenderContext> buildSenderContext(String emailTemplateKey, MailContext mailContext) {
        Option<EmailTemplateEntity> emailTemplate = emailTemplateDao.findByKeyO(emailTemplateKey);
        if (!emailTemplate.isPresent()) {
            logger.error("Unknown email template key: {}", emailTemplateKey);
            return Option.empty();
        }

        Option<String> language = mailContext.getLanguage().map(Language::value);
        if (!language.isPresent()) {
            language = getLanguage(mailContext.getTo());
            if (!language.isPresent()) {
                logger.error("Unable to determine language of user {}", mailContext.getTo());
                return Option.empty();
            }
        }

        Option<LocalizedEmailTemplateEntity> emailTemplateLocalized = resolveLocalizedTemplate(
                emailTemplateKey, language);

        if (!emailTemplateLocalized.isPresent()) {
            logger.warn("No localized email template for template {} and language {}", emailTemplateKey, language.get());
            return Option.empty();
        }
        mailContext.setLanguage(Language.R.fromValueO(emailTemplateLocalized.get().getLocale()));

        SenderContext.SenderContextBuilder builder;

        if (mailContext.getEmail().isPresent()) {
            builder = SenderContext.builder().email(mailContext.getEmail().get().getEmail());
        } else {
            builder = SenderContext.builder().to(mailContext.getTo());
        }
        SenderContext context = builder
                .args(buildArgs(mailContext, emailTemplate.get()))
                .senderTemplateDefinition(Utils.parseTemplateDefinition(emailTemplateLocalized.get().getSenderCode()))
                .build();

        return Option.of(context);
    }

    private Option<LocalizedEmailTemplateEntity> resolveLocalizedTemplate(
            String templateKey, Option<String> preferredLanguage
    ) {
        ListF<LocalizedEmailTemplateEntity> entries = emailTemplateDao.findLocalizations(templateKey);

        return getPrioritizesLanguages(preferredLanguage).iterator()
                .filterMap(l -> entries.find(e -> e.getLocale().equals(l)))
                .nextO();
    }

    private Option<String> getLanguage(PassportUid uid) {
        BlackboxCorrectResponse response = blackbox2.query()
                .userInfo(LocalhostUtils.localAddress(), uid, Cf.list(BlackboxDbFields.LANG), Cf.list());
        return response.getDbFields().getO(BlackboxDbFields.LANG);
    }

    private ListF<String> getPrioritizesLanguages(Option<String> preferred) {
        for (Tuple2<ListF<String>, ListF<String>> mapping : parseLanguageMappings(emailsLanguageMappings.get())) {
            ListF<String> from = mapping.get1(), to = mapping.get2();

            if (from.containsTs("*") || preferred.exists(from::containsTs)) {
                return to.containsTs("*") ? preferred.plus(to.filterNot("*"::equals)) : to;
            }
        }
        throw new IllegalStateException("No mapping found for language " + preferred.orElse((String) null));
    }

    private boolean validateLanguageMappings(String value) {
        try {
            Tuple2List<ListF<String>, ListF<String>> mappings = parseLanguageMappings(value);

            for (Tuple2<ListF<String>, ListF<String>> mapping : mappings) {
                for (ListF<String> list : Arrays.asList(mapping.get1(), mapping.get2())) {
                    Validate.notEmpty(list);
                    Validate.forAll(list, lang -> lang.matches("[a-z]{2}|\\*"));
                }
            }
            Validate.exists(mappings, m -> m.get1().containsTs("*"));
            return true;

        } catch (Exception e) {
            return false;
        }
    }

    private Tuple2List<ListF<String>, ListF<String>> parseLanguageMappings(String value) {
        Tuple2List<ListF<String>, ListF<String>> mappings = Tuple2List.arrayList();

        for (String mapping : value.split(" *; *")) {
            String[] split = mapping.trim().split(" *-> *");

            ListF<String> from = Cf.x(split[0].split(" *, *"));
            ListF<String> to = Cf.x(split[1].split(" *, *"));

            mappings.add(from, to);
        }
        return mappings;
    }

    private MapF<String, Object> buildArgs(MailContext mailContext, EmailTemplateEntity emailTemplate) {
        ListF<String> args = emailTemplate.getArgs();
        MapF<String, Object> argsWithValues = Cf.hashMap();
        for (String arg : args) {
            Option<EmailKeyDataProvider> keyProvider = keyProvidersMap.getO(arg);
            if (!keyProvider.isPresent()) {
                throw new NotImplementedException(String.format("key provider for argument %s is not found", arg));
            }
            Option<String> argValue = keyProvider.get().getKeyData(arg, mailContext);
            if (argValue.isPresent()) {
                argsWithValues.put(arg, argValue.get());
                continue;
            }
            logger.warn("email argument %s not handled for email template %s", arg, emailTemplate.getKey());
        }

        return argsWithValues;
    }

    private static MapF<String, EmailKeyDataProvider> createKeyProvidersMap(ListF<EmailKeyDataProvider> providers) {
        MapF<String, EmailKeyDataProvider> map = Cf.hashMap();
        for (EmailKeyDataProvider provider : providers) {
            for (String key : provider.getAcceptKeys()) {
                if (map.containsKeyTs(key)) {
                    throw new IllegalStateException("duplicate key provide for argument " + key);
                }

                map.put(key, provider);
            }
        }
        return map;
    }
}
