package ru.yandex.chemodan.app.psbilling.core.mail.dataproviders;

import java.util.UUID;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.mail.EventMailType;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderContext;
import ru.yandex.chemodan.app.psbilling.core.mail.localization.DateResolver;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.utils.Language;

import static ru.yandex.chemodan.app.psbilling.core.mail.Utils.DELIMITER;

public class SubscriptionCancelledSenderDataProvider extends AbstractPochtaProSenderDataProvider {

    private final DateResolver dateResolver;

    private final DynamicProperty<String> senderRuTemplateDefinition =
            new DynamicProperty<>("ps-billing-transactional-emails-pochta-subscription-cancelled-ru-template-definition",
                    "Yandex.Mail" + DELIMITER + "8YPXBSN3-WHQ");

    public SubscriptionCancelledSenderDataProvider(Blackbox2 blackbox2, GroupsManager groupsManager,
            UserServiceManager userServiceManager, DateResolver dateResolver)
    {
        super(blackbox2, groupsManager, userServiceManager);
        this.dateResolver = dateResolver;
    }

    @Override
    protected DynamicProperty<String> getRuSenderTemplate() {
        return senderRuTemplateDefinition;
    }

    @Override
    public Option<SenderContext> buildSenderContext(MailContext mailContext) {
        checkMailContextPreconditions(mailContext);

        PassportUid to = mailContext.getTo();
        Language language = mailContext.getLanguage().getOrThrow(IllegalStateException::new);
        UserService userService = mailContext.getUserServiceId()
                .map(UUID::fromString)
                .map(getUserServiceManager()::findById)
                .get();

        String publicDisplayName = getPublicNameForUid(to);

        String tariffName = getTariffName(userService, language);

        String subscriptionEndDate = userService.getNextCheckDate()
                .map(checkDate -> dateResolver.getLocalizedMskDate(checkDate, language))
                .getOrThrow(IllegalStateException::new);

        return Option.of(SenderContext.builder()
                .args(Cf.map(
                        "public_display_name", publicDisplayName,
                        "subscription_end_date", subscriptionEndDate,
                        "tariff_name", tariffName
                ))
                .attachments(Cf.list())
                .to(to)
                .build()
        );
    }

    @Override
    public boolean isAcceptableFor(EventMailType eventMailType, MailContext context) {
        return EventMailType.SUBSCRIPTION_CANCELLED.equals(eventMailType);
    }
}
