package ru.yandex.chemodan.app.psbilling.core.mail.dataproviders;

import java.util.UUID;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.mail.EventMailType;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderContext;
import ru.yandex.chemodan.app.psbilling.core.mail.localization.DateResolver;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPrice;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.utils.Language;

import static ru.yandex.chemodan.app.psbilling.core.mail.Utils.DELIMITER;

public class SubscriptionHasNotBeenPaidSenderDataProvider extends AbstractPochtaProSenderDataProvider {

    private final UserProductManager userProductManager;

    private final DateResolver dateResolver;

    private final DynamicProperty<String> senderRuTemplateDefinition =
            new DynamicProperty<>(
                    "ps-billing-transactional-emails-pochta-subscription-has-not-been-paid-ru-template-definition",
                    "Yandex.Mail" + DELIMITER + "0AXYESN3-DZC");

    public SubscriptionHasNotBeenPaidSenderDataProvider(Blackbox2 blackbox2, GroupsManager groupsManager,
            UserServiceManager userServiceManager, UserProductManager userProductManager, DateResolver dateResolver)
    {
        super(blackbox2, groupsManager, userServiceManager);
        this.userProductManager = userProductManager;
        this.dateResolver = dateResolver;
    }

    @Override
    protected DynamicProperty<String> getRuSenderTemplate() {
        return senderRuTemplateDefinition;
    }

    @Override
    public Option<SenderContext> buildSenderContext(MailContext mailContext) {
        checkMailContextPreconditions(mailContext);

        PassportUid to = mailContext.getTo();
        Language language = mailContext.getLanguage().getOrThrow(IllegalStateException::new);
        UserService userService = mailContext.getUserServiceId()
                .map(UUID::fromString)
                .map(getUserServiceManager()::findById)
                .get();

        String publicDisplayName = getPublicNameForUid(to);

        String tariffName = getTariffName(userService, language);

        UserProductPrice productPrice = userService.getPrice().getOrThrow(IllegalStateException::new);
        String sum = getMoneySum(productPrice.getPrice());
        String currency = productPrice.getCurrencyCode();

        String purchaseDeadline = userService.getSubscriptionUntil()
                .map(date -> dateResolver.getLocalizedMskDate(date, language))
                .getOrThrow(IllegalStateException::new);

        return Option.of(SenderContext.builder()
                .args(Cf.<String, Object>map(
                        "public_display_name", publicDisplayName,
                        "tariff_name", tariffName,
                        "sum", sum,
                        "currency", getCurrencyByCode(currency.toUpperCase())
                ).plus1("purchase_dedline", purchaseDeadline)) //the same typo as in the sender template
                .attachments(Cf.list())
                .to(to)
                .build()
        );
    }

    @Override
    public boolean isAcceptableFor(EventMailType eventMailType, MailContext context) {
        return EventMailType.SUBSCRIPTION_HAS_NOT_BEEN_PAID.equals(eventMailType);
    }
}
