package ru.yandex.chemodan.app.psbilling.core.mail.dataproviders;

import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServicePriceOverrideDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductOwnerDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.texts.TankerKeyDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupServicePriceOverride;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.PriceOverrideReason;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupsManager;
import ru.yandex.chemodan.app.psbilling.core.mail.EventMailType;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.dataproviders.model.SenderContext;
import ru.yandex.chemodan.app.psbilling.core.mail.localization.DateResolver;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.balanceclient.model.response.FindClientResponseItem;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.chemodan.app.psbilling.core.mail.Utils.DELIMITER;

public class TrialEndsSoonSenderDataProvider extends AbstractSenderByProductOwnerDataProvider {

    private final static Logger logger = LoggerFactory.getLogger(TrialEndsSoonSenderDataProvider.class);

    private final DateResolver dateResolver;

    private final GroupServicePriceOverrideDao groupServicePriceOverrideDao;

    private final GroupServiceDao groupServiceDao;

    private final DynamicProperty<String> mailRuSenderTemplateDefinition =
            new DynamicProperty<>("ps-billing-transactional-emails-mail-trial-ends-ru-template-definition",
                    "disk" + DELIMITER + "CFNBUSN3-76F1");

    private final DynamicProperty<String> diskRuSenderTemplateDefinition =
            new DynamicProperty<>("ps-billing-transactional-emails-disk-trial-ends-ru-template-definition",
                    "disk" + DELIMITER + "KNMBUSN3-UW3");

    private MapF<Language, DynamicProperty<String>> mailTemplatesByLanguage =
            Cf.map(Language.RUSSIAN, mailRuSenderTemplateDefinition);

    private MapF<Language, DynamicProperty<String>> diskTemplatesByLanguage =
            Cf.map(Language.RUSSIAN, diskRuSenderTemplateDefinition);

    public TrialEndsSoonSenderDataProvider(Blackbox2 blackbox2, GroupsManager groupsManager,
            ProductOwnerDao productOwnerDao, UserProductDao userProductDao,
            BalanceService balanceService, GroupDao groupDao, GroupProductDao groupProductDao,
            TankerKeyDao tankerKeyDao, DateResolver dateResolver,
            GroupServicePriceOverrideDao groupServicePriceOverrideDao,
            GroupServiceDao groupServiceDao)
    {
        super(blackbox2, groupsManager, productOwnerDao, userProductDao, balanceService,
                groupDao, groupProductDao, tankerKeyDao);
        this.dateResolver = dateResolver;
        this.groupServicePriceOverrideDao = groupServicePriceOverrideDao;
        this.groupServiceDao = groupServiceDao;
    }

    @Override
    protected MapF<Language, DynamicProperty<String>> getTemplatesByLanguageForDisk() {
        return diskTemplatesByLanguage;
    }

    @Override
    protected MapF<Language, DynamicProperty<String>> getTemplatesByLanguageForMail() {
        return mailTemplatesByLanguage;
    }

    @Override
    public Option<SenderContext> buildSenderContext(MailContext mailContext) {
        PassportUid to = mailContext.getTo();
        Language language = mailContext.getLanguage().getOrThrow(IllegalStateException::new);

        checkMailContextPreconditionsForOneGroupService(mailContext);
        UUID groupId = UUID.fromString(mailContext.getGroupIds().first());
        UUID groupServiceId = UUID.fromString(mailContext.getGroupServices().first().getGroupServiceId());

        GroupService groupService = groupServiceDao.findById(groupServiceId);
        if (!groupService.getTarget().equals(Target.ENABLED)) {
            logger.info("The service {} is not enabled", groupService);
            return Option.empty();
        }

        String publicDisplayName = getPublicNameForUid(to);

        String tariffName = getTariffName(groupServiceId, language);

        Group organization = getOrganizationById(groupId);
        FindClientResponseItem balancePaymentData = getBalanceClientDataForOrganization(organization);
        String organizationName = balancePaymentData.getName();

        Instant now = Instant.now();
        ListF<GroupServicePriceOverride> groupServicePriceOverrides =
                groupServicePriceOverrideDao.findByGroupServices(Cf.list(groupServiceId))
                        .getO(groupServiceId).getOrElse(Cf::list);
        GroupServicePriceOverride groupServicePriceOverride =
                groupServicePriceOverrides
                        .filter(priceOverride -> PriceOverrideReason.TRIAL.equals(priceOverride.getReason()))
                        .find(priceOverride -> priceOverride.getEndDate().map(now::isBefore).getOrElse(Boolean.FALSE))
                        .getOrThrow(IllegalStateException::new);
        if (groupServicePriceOverride.isHidden()) {
            logger.info("The trial is hidden {}", groupServicePriceOverride);
            return Option.empty();
        }
        String trialEndDate = dateResolver.getLocalizedMskDate(groupServicePriceOverride.getEndDate().get(), language);

        GroupProductEntity groupProductEntity = getGroupProductDao().findByGroupServiceId(groupServiceId);
        String sumPerMonth = getMoneySum(groupProductEntity.getPricePerUserInMonth());
        String currency = groupProductEntity.getPriceCurrency().getCurrencyCode();

        String paymentStartDate = dateResolver.getLocalizedMskDate(
                getFirstPaidDate(groupService, groupServicePriceOverrides),
                language
        );

        return Option.of(SenderContext.builder()
                .args(Cf.<String, Object>map(
                        "public_display_name", publicDisplayName,
                        "tariff_name", tariffName,
                        "org_name", organizationName,
                        "trial_end_date", trialEndDate
                ).plus1("payment_start_date", paymentStartDate)
                        .plus1("sum", sumPerMonth)
                        .plus1("currency", getCurrencyByCode(currency.toUpperCase())))
                .attachments(Cf.list())
                .to(to)
                .email(getEmail(balancePaymentData))
                .build()
        );
    }

    @Override
    public boolean isAcceptableFor(EventMailType eventMailType, MailContext context) {
        return EventMailType.TRIAL_WILL_ENDS_SOON.equals(eventMailType);
    }

    private Instant getFirstPaidDate(GroupService groupService,
            ListF<GroupServicePriceOverride> groupServicePriceOverrides)
    {
        return getGroupsManager().findFirstPaidDay(groupService, groupServicePriceOverrides);
    }
}
