package ru.yandex.chemodan.app.psbilling.core.mail.keydataproviders;

import java.util.UUID;
import java.util.function.Function;

import org.joda.time.Instant;
import org.joda.time.format.DateTimeFormat;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductDao;
import ru.yandex.chemodan.app.psbilling.core.mail.GroupServiceData;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.mail.localization.DateResolver;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.utils.Language;

public abstract class BaseEmailKeyDataProvider implements EmailKeyDataProvider {
    private MapF<String, Function<MailContext, Option<String>>> handlers;

    private synchronized MapF<String, Function<MailContext, Option<String>>> getHandlers() {
        if (handlers == null) {
            handlers = constructHandlers();
        }
        return handlers;
    }

    protected abstract MapF<String, Function<MailContext, Option<String>>> constructHandlers();

    @Override
    public ListF<String> getAcceptKeys() {
        return getHandlers().keys();
    }

    @Override
    public Option<String> getKeyData(String key, MailContext context) {
        if (!getAcceptKeys().containsTs(key)) {
            return Option.empty();
        }
        return getHandlers().getTs(key).apply(context);
    }

    protected Option<String> formatDate(
            DateResolver dateResolver, Option<Instant> dateO, PassportUid uid, Option<Language> language) {
        if (!dateO.isPresent()) {
            return Option.of("");
        }
        Instant date = dateO.get();

        if (language.isPresent()) {
            return Option.of(dateResolver.getLocalizedUserDate(date, uid, language.get()));
        }

        return Option.of(date.toString(DateTimeFormat.forPattern("dd.MM.yyyy")));
    }

    protected ListF<UserProduct> getUserProductsByUserProductId(UserProductManager userProductManager, UUID userProductId) {
        return Cf.list(userProductManager.findById(userProductId));
    }

    protected ListF<UserProduct> getUserProductsByGroupServiceIds(UserProductManager userProductManager,
                                                                  UserProductDao userProductDao,
                                                                  ListF<GroupServiceData> groupServices) {
        return groupServices
                .map(svc -> userProductDao.findByGroupServiceId(UUID.fromString(svc.getGroupServiceId())))
                .map(prod -> userProductManager.findById(prod.getId()));
    }
}
