package ru.yandex.chemodan.app.psbilling.core.mail.keydataproviders;

import java.util.function.Function;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductDao;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductFeature;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductFeatureRegistry;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.inside.utils.Language;

@AllArgsConstructor
public class FanMailLimitKeyDataProvider extends BaseEmailKeyDataProvider {

    private final UserProductManager userProductManager;
    private final UserProductDao userProductDao;

    @Override
    protected MapF<String, Function<MailContext, Option<String>>> constructHandlers() {
        return Cf.map("fan_mail_limit_product", this::getFanMailLimitByUserProduct,
                "fan_mail_limit_service", this::getFanMailLimitByService);
    }

    private Option<String> getFanMailLimitByUserProduct(MailContext context) {
        if (context.getUserProductId().isEmpty()) {
            return Option.empty();
        }
        ListF<UserProduct> userProducts = getUserProductsByUserProductId(userProductManager,
                context.getUserProductId().get());
        return Option.of(getFeatureValue(userProducts, context));
    }

    private Option<String> getFanMailLimitByService(MailContext context) {
        if (context.getGroupServices().isEmpty()) {
            return Option.empty();
        }
        ListF<UserProduct> userProducts = getUserProductsByGroupServiceIds(userProductManager, userProductDao,
                context.getGroupServices());
        return Option.of(getFeatureValue(userProducts, context));
    }

    private String getFeatureValue(ListF<UserProduct> userProducts, MailContext context) {
        Language lang = context.getLanguage()
                .orElseThrow(() -> new IllegalStateException("Missing language"));

        UserProductFeature feature = userProductManager.getEnabledFeature(userProducts,
                        productFeature -> UserProductFeatureRegistry.isFanMailLimitFeature(productFeature.getCode()))
                .orElseThrow(() -> new IllegalStateException("Missing fan mail limit"));

        return feature.getValue()
                .flatMapO(t -> t.findLanguageTranslation(lang.value()))
                .orElseThrow(() -> new IllegalStateException("Missing fan mail limit value"));
    }
}
