package ru.yandex.chemodan.app.psbilling.core.mail.keydataproviders;

import java.util.NoSuchElementException;
import java.util.function.Function;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductDao;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductFeature;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductFeatureRegistry;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.texts.PredefinedTankerKey;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;

@AllArgsConstructor
public class MailArchiveTextKeyDataProvider extends BaseEmailKeyDataProvider {
    private final UserProductDao userProductDao;
    private final UserProductManager userProductManager;
    private final TextsManager textsManager;

    @Override
    protected MapF<String, Function<MailContext, Option<String>>> constructHandlers() {
        return Cf.map("mail_archive_text_product", this::getMailArchiveTextByUserProduct,
                "mail_archive_text_service", this::getMailArchiveTextByService,
                "is_mail_archive_switched_on", this::getIsMailArchiveSwitchedOn);
    }

    private Option<String> getIsMailArchiveSwitchedOn(MailContext context) {
        if (context.getUserProductId().isEmpty() || context.getOldUserProductId().isEmpty()) {
            return Option.empty();
        }
        ListF<UserProduct> userProducts =
                getUserProductsByUserProductId(userProductManager, context.getUserProductId().get());
        ListF<UserProduct> oldUserProducts =
                getUserProductsByUserProductId(userProductManager, context.getOldUserProductId().get());

        boolean isFeatureEnabled = userProductManager.getEnabledFeature(userProducts,
                        productFeature -> UserProductFeatureRegistry.isLetterArchiveAvailableFeature(productFeature.getCode()))
                .isPresent();
        boolean isOldFeatureEnabled = userProductManager.getEnabledFeature(oldUserProducts,
                        productFeature -> UserProductFeatureRegistry.isLetterArchiveAvailableFeature(productFeature.getCode()))
                .isPresent();
        return Option.of(Boolean.toString(isFeatureEnabled && !isOldFeatureEnabled));
    }

    private Option<String> getMailArchiveTextByUserProduct(MailContext context) {
        if (context.getUserProductId().isEmpty()) {
            return Option.empty();
        }
        ListF<UserProduct> userProducts = getUserProductsByUserProductId(userProductManager,
                context.getUserProductId().get());
        return getFeatureValue(userProducts, context);
    }

    private Option<String> getMailArchiveTextByService(MailContext context) {
        if (context.getGroupServices().isEmpty()) {
            return Option.empty();
        }
        ListF<UserProduct> userProducts = getUserProductsByGroupServiceIds(userProductManager, userProductDao,
                context.getGroupServices());
        return getFeatureValue(userProducts, context);
    }

    private Option<String> getFeatureValue(ListF<UserProduct> userProducts, MailContext context) {
        Option<UserProductFeature> feature = userProductManager.getEnabledFeature(userProducts,
                productFeature -> UserProductFeatureRegistry.isLetterArchiveAvailableFeature(productFeature.getCode()));
        if (feature.isEmpty() || !feature.get().isEnabled()) {
            return Option.of("");
        }
        String language = context.getLanguage().get().value();
        String translation = textsManager.findTranslation(
                PredefinedTankerKey.MailArchiveText.getTankerProject(),
                PredefinedTankerKey.MailArchiveText.getTankerKeySet(),
                PredefinedTankerKey.MailArchiveText.getTankerKey()
        )
                .findLanguageTranslation(language)
                .orElseThrow(() -> new NoSuchElementException("No translation mail_archive_text_in_email to " + language));
        return Option.of(translation);
    }
}
