package ru.yandex.chemodan.app.psbilling.core.mail.keydataproviders;

import java.util.UUID;
import java.util.function.Function;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.texts.TankerKeyDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.texts.TankerTranslationEntity;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.texts.TankerTranslation;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.inside.utils.Language;

@AllArgsConstructor
public class TariffNameKeyDataProvider extends BaseEmailKeyDataProvider {
    private final UserServiceManager userServiceManager;
    private final UserProductManager userProductManager;
    private final GroupProductDao groupProductDao;
    private final TankerKeyDao tankerKeyDao;

    @Override
    protected MapF<String, Function<MailContext, Option<String>>> constructHandlers() {
        return Cf.map("tariff_name", this::getTariffName,
                "tariff_name_old", this::getOldTariffName);
    }

    private Option<String> getTariffName(MailContext context) {
        return getUserTariffName(context)
                .orElse(getGroupTariffName(context))
                .orElse(getUserProductTariffName(context));
    }

    private Option<String> getOldTariffName(MailContext context) {
        if (!context.getOldUserProductId().isPresent()) {
            return Option.empty();
        }
        return getTariffNameByUserProductId(context, context.getOldUserProductId().get());
    }

    private Option<String> getUserTariffName(MailContext context) {
        if (!context.getUserServiceId().isPresent()) {
            return Option.empty();
        }
        Language language = context.getLanguage().getOrThrow(IllegalStateException::new);
        UserService userService = context.getUserServiceId()
                .map(UUID::fromString)
                .map(userServiceManager::findById)
                .get();
        UserProduct product = userService.getUserProduct();
        return getUserProductTariffName(product, language);
    }

    private Option<String> getGroupTariffName(MailContext context) {
        if (!context.getGroupServices().isNotEmpty()) {
            return Option.empty();
        }
        Language language = context.getLanguage().getOrThrow(IllegalStateException::new);
        UUID groupServiceId = UUID.fromString(context.getGroupServices().first().getGroupServiceId());
        GroupProductEntity groupProductEntity = groupProductDao.findByGroupServiceId(groupServiceId);
        ListF<TankerTranslationEntity> tankerTranslations =
                tankerKeyDao.findTranslations(groupProductEntity.getTitleTankerKeyId()
                        .getOrThrow("No tanker data for group product"));
        return Option.of(
                tankerTranslations.filter(tankerTranslation -> tankerTranslation.getLocale().equals(language.value()))
                        .first().getTranslationText());
    }

    private Option<String> getUserProductTariffName(MailContext context) {
        if (!context.getUserProductId().isPresent()) {
            return Option.empty();
        }
        return getTariffNameByUserProductId(context, context.getUserProductId().get());
    }

    private Option<String> getTariffNameByUserProductId(MailContext context, UUID userProductId) {
        Language language = context.getLanguage().getOrThrow(IllegalStateException::new);
        UserProduct product = userProductManager.findById(userProductId);
        return getUserProductTariffName(product, language);
    }

    private Option<String> getUserProductTariffName(UserProduct product, Language language) {
        TankerTranslation title = product.getTitle().getOrThrow("No tanker data for user product");

        String translation = title.findLanguageTranslation(language.value())
                .getOrThrow("No translation " + product.getCode() + " title to " + language);
        return Option.of(translation);
    }
}
