package ru.yandex.chemodan.app.psbilling.core.mail.localization;

import java.io.IOException;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.time.MoscowTime;

@Slf4j
public class DateResolver {

    private final DatesLocalizationData datesLocalizationData;
    private final UserTimezoneHelper userTimezoneHelper;

    public DateResolver(ObjectMapper objectMapper, UserTimezoneHelper userTimezoneHelper) throws IOException {
        datesLocalizationData = objectMapper.readValue(
                new ClassPathResourceInputStreamSource(DateResolver.class, "dates.json").getInput(),
                DatesLocalizationData.class);
        this.userTimezoneHelper = userTimezoneHelper;
    }

    public String getLocalizedMskDate(Instant instant, Language language) {
        LocalDate date = new LocalDate(instant, MoscowTime.TZ);
        return getLocalizedDate(date, language);
    }

    public String getLocalizedUserDate(Instant instant, PassportUid uid, Language language) {
        DateTimeZone userTimezone = userTimezoneHelper.getUserTimezone(uid);
        LocalDate date = new LocalDate(instant, userTimezone);
        return getLocalizedDate(date, language);
    }

    public String getLocalizedDate(LocalDate date, Language language) {
        String month = datesLocalizationData.getMonths().get(date.getMonthOfYear() - 1)
                .getOrThrow(
                        language.value(),
                        String.format("Cannot find month definition for date %s and language %s", date,
                                language.value()));
        String yearTerm = datesLocalizationData.getYearTerm().getOrThrow(
                language.value(),
                String.format("Cannot find the year term for language %s", language.value())
        );
        return datesLocalizationData.getDateFormats().getOrThrow(language.value(),
                        String.format("Cannot find date format for language %s", language.value()))
                .replaceAll("\\{\\{day}}", String.valueOf(date.getDayOfMonth()))
                .replaceAll("\\{\\{month}}", month)
                .replaceAll("\\{\\{year}}", String.valueOf(date.getYear()))
                .replaceAll("\\{\\{yearTerm}}", yearTerm);
    }
}
