package ru.yandex.chemodan.app.psbilling.core.mail.localization;

import java.io.IOException;
import java.util.Comparator;
import java.util.Map;
import java.util.TreeMap;

import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.lang.StringUtils;

public class TermsResolver {

    private final TermsLocalizationData termsLocalizationData;

    public TermsResolver(ObjectMapper objectMapper) throws IOException {
        termsLocalizationData = objectMapper.readValue(
                new ClassPathResourceInputStreamSource(TermsResolver.class, "terms.json").getInput(),
                TermsLocalizationData.class);
    }

    public String getOneFormOfTerm(String term, Language language) {
        return termsLocalizationData.getData()
                .getOrThrow(term, () -> String.format("Cannot find term %s", term))
                .getOneForm()
                .getOrThrow(language.value(), () -> String.format("Cannot find the one form of %s for language %s",
                        term, language));
    }

    public String getTermForCustomPeriodPeriod(CustomPeriod period, Language language) {
        if (period.getValue() == 1) {
            return getOneFormOfTerm(period.getUnit().value(), language);
        }
        return getManyFormOfTerm(period.getUnit().value(), language, period.getValue());
    }

    public String getManyFormOfTerm(String term, Language language, int howMany) {
        if (howMany == 1) {
            return getOneFormOfTerm(term, language);
        }
        ListF<TermManyForm> manyForms = termsLocalizationData.getData()
                .getOrThrow(term, () -> String.format("Cannot find term %s", term))
                .getManyForm().getOrThrow(language.value(), () -> String.format("Cannot find the many form of %s for " +
                        "language %s", term, language));
        TreeMap<Integer, String> sortedByFromReversed = new TreeMap<>(Comparator.<Integer>naturalOrder().reversed());
        manyForms.forEach(mf -> sortedByFromReversed.put(mf.getFrom(), mf.getForm()));

        for (Map.Entry<Integer, String> e : sortedByFromReversed.entrySet()) {
            if (howMany >= e.getKey()) {
                return parameterizeMsg(e.getValue(), howMany);
            }
        }
        throw new IllegalArgumentException("Can not find many form for term " + term + " number " + howMany + " in " +
                "language " + language);
    }

    private String parameterizeMsg(String localized, int howMany) {
        return StringUtils.replace(localized, "{{count}}", Integer.toString(howMany));
    }
}
