package ru.yandex.chemodan.app.psbilling.core.mail.tasks;

import java.math.BigDecimal;
import java.util.Set;
import java.util.UUID;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupTrustPaymentRequestDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.GroupTrustPaymentRequest;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.billing.PaymentRequestStatus;
import ru.yandex.chemodan.app.psbilling.core.groups.GroupServicesManager;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.tasks.BaseTask;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskExecutor;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.tasks.policies.TaskPreExecutionPoliciesHolder;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

public class B2BAdminAbandonedPaymentTask extends BaseTask<B2BAdminAbandonedPaymentTask.Parameters> {
    private static final Set<PaymentRequestStatus> PAYMENT_STATUSES_TO_RESCHEDULE_TASK =
            Cf.set(PaymentRequestStatus.INIT, PaymentRequestStatus.SUCCESS);
    private TaskScheduler taskScheduler;
    private GroupTrustPaymentRequestDao groupTrustPaymentRequestDao;
    private GroupServicesManager groupServicesManager;
    private GroupProductDao groupProductDao;

    public B2BAdminAbandonedPaymentTask(Parameters parameters) {
        super(parameters);
    }

    public B2BAdminAbandonedPaymentTask(TaskPreExecutionPoliciesHolder taskPreExecutionPoliciesHolder,
                                        ListF<TaskExecutor<?>> taskExecutors, TaskScheduler taskScheduler,
                                        GroupTrustPaymentRequestDao groupTrustPaymentRequestDao,
                                        GroupServicesManager groupServicesManager, GroupProductDao groupProductDao) {
        super(taskPreExecutionPoliciesHolder, taskExecutors, B2BAdminAbandonedPaymentTask.Parameters.class);
        this.taskScheduler = taskScheduler;
        this.groupTrustPaymentRequestDao = groupTrustPaymentRequestDao;
        this.groupServicesManager = groupServicesManager;
        this.groupProductDao = groupProductDao;
    }

    @Override
    protected void execute(Parameters taskParams, ExecutionContext context) {
        PassportUid uid = taskParams.getContext().getTo();
        UUID userProductId = taskParams.getContext().getUserProductId().get();
        UUID groupId = taskParams.groupId;
        long clientId = taskParams.clientId;
        Duration lookupTime = taskParams.lookupTime;
        ListF<GroupTrustPaymentRequest> paymentRequests = groupTrustPaymentRequestDao.findRecentPayments(
                taskParams.clientId, Option.of(Instant.now().minus(lookupTime)));
        boolean noPaidSubscription = groupServicesManager.find(groupId, Target.ENABLED)
                .map(GroupService::getGroupProductId)
                .map(groupProductDao::findById)
                .map(GroupProductEntity::getPricePerUserInMonth)
                .filter(price -> price.compareTo(BigDecimal.ZERO) > 0)
                .isEmpty();
        if (!noPaidSubscription) {
            logger.info("Found paid subscription. Cancelling email to %d", uid.getUid());
            return;
        }
        if (paymentRequests.stream().anyMatch(
                paymentRequest -> PAYMENT_STATUSES_TO_RESCHEDULE_TASK.contains(paymentRequest.getStatus()))) {
            logger.info("Found ongoing payment request. Rescheduling email to %d to later times", uid.getUid());
            taskScheduler.scheduleB2bAdminAbandonedPaymentTask(uid, clientId, userProductId, groupId);
            return;
        }

        super.execute(taskParams, context);
    }

    @BenderBindAllFields
    public static class Parameters extends BaseEmailTask.Parameters {
        private final long clientId;
        private final UUID groupId;
        private final Duration lookupTime;

        public Parameters(long clientId, UUID groupId, Duration lookupTime, String emailKey, MailContext context,
                          String preExecutionPolicy, String executorKey, Option<Duration> cooldown) {
            super(emailKey, context, preExecutionPolicy, executorKey, cooldown);
            this.clientId = clientId;
            this.groupId = groupId;
            this.lookupTime = lookupTime;
        }
    }
}

