package ru.yandex.chemodan.app.psbilling.core.mail.tasks;

import java.util.UUID;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.balance.BalanceService;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.BalancePaymentInfo;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.synchronization.PsBillingQueueNames;
import ru.yandex.chemodan.app.psbilling.core.tasks.BaseTask;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskExecutor;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.tasks.policies.TaskPreExecutionPoliciesHolder;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.Blackbox2;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.BlackboxSid;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.net.LocalhostUtils;

public class EmailPaymentsAwareMembersTask extends BaseTask<EmailPaymentsAwareMembersTask.Parameters> {

    private GroupDao groupDao;
    private TaskScheduler taskScheduler;
    private BalanceService balanceService;
    private Blackbox2 blackbox2;

    public EmailPaymentsAwareMembersTask(Parameters parameters) {
        super(parameters);
    }

    public EmailPaymentsAwareMembersTask(
            TaskPreExecutionPoliciesHolder taskPreExecutionPoliciesHolder,
            ListF<TaskExecutor<?>> taskExecutors,
            GroupDao groupDao,
            TaskScheduler taskScheduler,
            BalanceService balanceService,
            Blackbox2 blackbox2
    ) {
        super(taskPreExecutionPoliciesHolder, taskExecutors, EmailPaymentsAwareMembersTask.Parameters.class);
        this.groupDao = groupDao;
        this.taskScheduler = taskScheduler;
        this.balanceService = balanceService;
        this.blackbox2 = blackbox2;
    }

    @Override
    protected void execute(Parameters parameters, ExecutionContext context) {
        Group group = groupDao.findById(parameters.groupId);

        PassportUid ownerUid = group.getOwnerUid();

        Option<PassportUid> payerUid = group.getPaymentInfo()
                .map(BalancePaymentInfo::getPassportUid);

        Option<Email> paymentEmail = group.getPaymentInfo()
                .flatMapO(info -> balanceService.findActiveContractPaymentEmail(info.getClientId()));

        Option<PassportUid> paymentEmailUid = paymentEmail.filterMap(this::getUidByEmail);

        schedule(parameters, ownerUid, Option.empty());

        if (payerUid.isPresent() && !payerUid.isSome(ownerUid)) {
            schedule(parameters, payerUid.get(), Option.empty());
        }
        if (paymentEmail.isPresent()
                && !paymentEmailUid.isSome(ownerUid)
                && !paymentEmailUid.exists(payerUid::isSome)) {
            schedule(parameters, paymentEmailUid.orElse(payerUid).orElse(ownerUid), paymentEmail);
        }
    }

    private void schedule(Parameters parameters, PassportUid to, Option<Email> email) {
        taskScheduler.scheduleLocalizedEmailTask(
                parameters.getEmailKey(),
                parameters.getContext().withTo(to).withEmail(email),
                parameters.getCooldown()
        );
    }

    private Option<PassportUid> getUidByEmail(Email email) {
        BlackboxCorrectResponse response = blackbox2.query()
                .userInfo(LocalhostUtils.localAddress(), email.getEmail(), BlackboxSid.SMTP, Cf.list());
        return response.getUid();
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(5);
    }

    @Override
    public TaskQueueName queueName() {
        return PsBillingQueueNames.REGULAR;
    }

    @BenderBindAllFields
    public static class Parameters extends BaseEmailTask.Parameters {
        private final UUID groupId;

        public Parameters(UUID groupId, String emailKey, MailContext context,
                          String preExecutionPolicy, String executorKey, Option<Duration> cooldown) {
            super(emailKey, context, preExecutionPolicy, executorKey, cooldown);
            this.groupId = groupId;
        }
    }
}
