package ru.yandex.chemodan.app.psbilling.core.products;

import java.util.UUID;

import lombok.AllArgsConstructor;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductLineDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductSetDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductLineEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductSetEntity;
import ru.yandex.chemodan.app.psbilling.core.products.selectors.DefaultAvailableSelector;
import ru.yandex.chemodan.app.psbilling.core.products.selectors.DefaultUnavailableSelector;
import ru.yandex.chemodan.app.psbilling.core.products.selectors.ProductLineAvailability;
import ru.yandex.chemodan.app.psbilling.core.products.selectors.ProductLineSelector;
import ru.yandex.chemodan.app.psbilling.core.products.selectors.SelectionContext;
import ru.yandex.chemodan.app.psbilling.core.promos.PromoService;
import ru.yandex.chemodan.app.psbilling.core.promos.PromoTemplate;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

@AllArgsConstructor
public class AbstractProductManager {
    private static final Logger logger = LoggerFactory.getLogger(AbstractProductManager.class);

    protected final ProductSetDao productSetDao;
    protected final ProductLineDao productLineDao;
    protected final PromoService promoService;
    private final SpringExpressionEvaluator springExpressionEvaluator;

    protected Option<LimitedTimeProductLine> findProductLine(String productSetKey, Option<PassportUid> uidO) {
        return findProductLine(productSetKey, uidO, true);
    }

    protected Option<LimitedTimeProductLine> findProductLine(String productSetKey, Option<PassportUid> uidO,
                                                             boolean includePromoLines) {
        Option<ProductSetEntity> productSetO = productSetDao.findByKey(productSetKey);
        if (!productSetO.isPresent()) {
            throw new A3ExceptionWithStatus(
                    "unknown_productset", "Product set '" + productSetKey + "' not found", HttpStatus.SC_404_NOT_FOUND);
        }

        return selectProductLine(productSetO.get().getId(), uidO, includePromoLines);
    }

    protected Option<LimitedTimeProductLine> selectProductLine(UUID productSetId, Option<PassportUid> uidO) {
        return selectProductLine(productSetId, uidO, true);
    }

    protected Option<LimitedTimeProductLine> selectProductLine(UUID productSetId, Option<PassportUid> uidO,
                                                               boolean includePromoLines) {
        ListF<ProductLineEntity> productLines = productLineDao.findByProductSetId(productSetId);
        return selectProductLine(productLines, uidO, includePromoLines);
    }

    protected Option<LimitedTimeProductLine> selectProductLine(ListF<ProductLineEntity> productLines,
                                                               Option<PassportUid> uidO, boolean includePromoLines) {
        MapF<UUID, PromoTemplate> promoLines = getProductLinesWithPromo(uidO);
        if (includePromoLines) {
            productLines = productLines.sorted(ProductLineEntity.topWithPromoThenByOrderNum(promoLines));
        } else {
            productLines = productLines.filter(line -> !promoLines.containsKeyTs(line.getId()))
                    .sortedBy(ProductLineEntity::getOrderNum);
        }

        SelectionContext context = new SelectionContext();
        for (ProductLineEntity productLine : productLines) {
            Option<PromoTemplate> productLinePromoO = promoLines.getO(productLine.getId());
            ProductLineAvailability availability = productLinePromoO.isNotEmpty()
                    ? testPromoProductLineAvailability(productLinePromoO.get(), productLine, context, uidO,
                    Option.empty())
                    : testProductLineAvailability(productLine, context, uidO, Option.empty());

            if (availability.isAvailable()) {
                logger.info("defined product line for user {}: {}", uidO, productLine);
                return Option.of(new LimitedTimeProductLine(
                        productLine, availability.getAvailableUntil(), productLinePromoO));
            }
        }

        logger.info("none of product lines {} is appreciable", productLines);
        return Option.empty();
    }

    protected ProductLineSelector evaluateSelector(ProductLineEntity line) {
        if (StringUtils.isEmpty(line.getSelectorBeanEL())) {
            return DefaultAvailableSelector.INSTANCE;
        }

        try {
            return springExpressionEvaluator.evaluateExpression(line.getSelectorBeanEL(), ProductLineSelector.class);
        } catch (Exception e) {
            logger.error("Error during evaluating selector bean for line {}. line will be disabled", line, e);
            return DefaultUnavailableSelector.INSTANCE;
        }
    }

    protected ProductLineAvailability testProductLineAvailability(
            ProductLineEntity line, SelectionContext context, Option<PassportUid> uidO, Option<Group> group) {
        ProductLineSelector selector = evaluateSelector(line);
        return selector.isAvailable(line, uidO, group, context);
    }


    protected ProductLineAvailability testPromoProductLineAvailability(PromoTemplate promoTemplate,
                                                                       ProductLineEntity line,
                                                                       SelectionContext context,
                                                                       Option<PassportUid> uid, Option<Group> group) {
        ProductLineAvailability lineAvailability = testProductLineAvailability(line, context, uid, group);
        if (!lineAvailability.isAvailable()) {
            return lineAvailability;
        }

        Option<Instant> promoToDateO = promoTemplate.canBeUsedUntilDate(uid);
        if (!promoToDateO.isPresent()) {
            return lineAvailability;
        }
        Option<Instant> availableUntil = lineAvailability.getAvailableUntil();
        if (!availableUntil.isPresent() || promoToDateO.get().isBefore(availableUntil.get())) {
            return ProductLineAvailability.availableUntil(promoToDateO.get());
        }

        return lineAvailability;

    }

    private MapF<UUID, PromoTemplate> getProductLinesWithPromo(Option<PassportUid> uidO) {
        return promoService.getReadyForUsingPromos(uidO)
                .flatMap(p -> p.getProductLineIds().map(l -> Tuple2.tuple(l, p)))
                .groupByMapValues(Tuple2::get1, Tuple2::get2)
                .mapValues(p -> p.min(PromoTemplate.topMultipleTimeThenByUntilDate(uidO)));
    }

    @Data
    protected static class LimitedTimeProductLine {
        private final ProductLineEntity productLine;
        private final Option<Instant> availableUntil;
        private final Option<PromoTemplate> promo;
    }
}
