package ru.yandex.chemodan.app.psbilling.core.products;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.UUID;

import lombok.EqualsAndHashCode;
import lombok.ToString;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.TrialDefinitionDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupPaymentType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialDefinitionEntity;
import ru.yandex.chemodan.app.psbilling.core.texts.TankerTranslation;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;

@EqualsAndHashCode(onlyExplicitlyIncluded = true)
@ToString(onlyExplicitlyIncluded = true)
public class GroupProduct {
    @EqualsAndHashCode.Include
    @ToString.Include
    private final GroupProductEntity productEntity;
    private final Function0<Option<TankerTranslation>> titleProvider;
    private final Function0<UserProduct> userProductProvider;
    private final Function0<Option<TrialDefinition>> trialDefinitionProvider;
    private final Function0<ListF<GroupProduct>> availableAddonsProvider;
    private final Function0<ListF<GroupProduct>> eligibleMainsProvider;

    public GroupProduct(GroupProductEntity productEntity, TextsManager textsManager,
                        UserProductManager userProductManager, TrialDefinitionDao trialDefinitionDao,
                        GroupProductDao groupProductDao)
    {
        this(productEntity,
                () -> productEntity.getTitleTankerKeyId().map(textsManager::findTranslation),
                () -> userProductManager.findById(productEntity.getUserProductId()),
                () -> productEntity.getTrialDefinitionId().map(trialDefinitionDao::findById).map(TrialDefinition::new),
                () -> groupProductDao.findAvailableAddons(productEntity.getId()).map(e -> new GroupProduct(e, textsManager, userProductManager, trialDefinitionDao, groupProductDao)),
                () -> groupProductDao.findEligibleMainProducts(productEntity.getId()).map(e -> new GroupProduct(e, textsManager, userProductManager, trialDefinitionDao, groupProductDao))
        );
    }

    public GroupProduct(GroupProductEntity productEntity, Option<TankerTranslation> title,
                        UserProduct userProduct, Option<TrialDefinitionEntity> trialDefinition,
                        Function0<ListF<GroupProduct>> availableAddonsProvider,
                        Function0<ListF<GroupProduct>> eligibleMainsProvider)
    {
        this(productEntity,
                () -> title,
                () -> userProduct,
                () -> trialDefinition.map(TrialDefinition::new),
                availableAddonsProvider,
                eligibleMainsProvider
        );
    }

    private GroupProduct(GroupProductEntity productEntity, Function0<Option<TankerTranslation>> titleProvider,
            Function0<UserProduct> userProductProvider,
            Function0<Option<TrialDefinition>> trialDefinitionProvider,
            Function0<ListF<GroupProduct>> availableAddonsProvider,
            Function0<ListF<GroupProduct>> eligibleMainsProvider)
    {
        this.productEntity = productEntity;
        this.titleProvider = titleProvider.memoize();
        this.userProductProvider = userProductProvider.memoize();
        this.trialDefinitionProvider = trialDefinitionProvider.memoize();
        this.availableAddonsProvider = availableAddonsProvider.memoize();
        this.eligibleMainsProvider = eligibleMainsProvider.memoize();
    }

    public boolean isSingleton() {
        return productEntity.isSingleton();
    }

    public Option<TankerTranslation> getTitle() {
        return titleProvider.apply();
    }

    public Option<UUID> getTrialDefinitionId() {
        return productEntity.getTrialDefinitionId();
    }

    public Option<TrialDefinition> getTrialDefinition() {
        return trialDefinitionProvider.apply();
    }

    public UUID getId() {
        return productEntity.getId();
    }

    public UUID getUserProductId() {
        return productEntity.getUserProductId();
    }

    public Currency getPriceCurrency() {
        return productEntity.getPriceCurrency();
    }

    public Option<Instant> getAvailableTo() {
        return productEntity.getAvailableTo();
    }

    public BigDecimal getPricePerUserInMonth() {
        return productEntity.getPricePerUserInMonth();
    }

    public Option<BigDecimal> getDisplayOriginalPrice() {
        return productEntity.getDisplayOriginalPrice();
    }

    public Option<BigDecimal> getDisplayDiscountPercent() {
        return productEntity.getDisplayDiscountPercent();
    }

    public boolean isSkipTransactionsExport() {
        return productEntity.isSkipTransactionsExport();
    }

    public boolean isHidden() {
        return productEntity.isHidden();
    }

    public UserProduct getUserProduct() {
        return userProductProvider.apply();
    }

    public String getCode() {
        return productEntity.getCode();
    }

    public ListF<UserProductFeature> getFeatures() {
        return getUserProduct().getFeatures();
    }

    public boolean isBestOffer() {
        return productEntity.isBestOffer();
    }

    public GroupPaymentType getPaymentType() {
        return productEntity.getPaymentType();
    }

    public boolean isPrepaid() {
        return productEntity.getPaymentType() == GroupPaymentType.PREPAID;
    }

    public boolean isFree() {
        return productEntity.getPricePerUserInMonth().equals(BigDecimal.ZERO);
    }

    public ListF<GroupProduct> getEligibleMainProducts() {
        return eligibleMainsProvider.apply();
    }

    public ListF<GroupProduct> getAvailableAddons() {
        return availableAddonsProvider.apply();
    }

    public GroupProductType getProductType() {
        return productEntity.getProductType();
    }
}
