package ru.yandex.chemodan.app.psbilling.core.products;

import java.util.Comparator;
import java.util.UUID;

import org.jetbrains.annotations.NotNull;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.config.featureflags.FeatureFlags;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.TrialDefinitionDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductLineDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductSetDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductType;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialDefinitionEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductLineEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductSetEntity;
import ru.yandex.chemodan.app.psbilling.core.products.selectors.ProductLineAvailability;
import ru.yandex.chemodan.app.psbilling.core.products.selectors.SelectionContext;
import ru.yandex.chemodan.app.psbilling.core.promos.PromoService;
import ru.yandex.chemodan.app.psbilling.core.promos.groups.AbstractGroupPromoTemplate;
import ru.yandex.chemodan.app.psbilling.core.promos.v2.GroupPromoService;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.chemodan.app.psbilling.core.texts.TankerTranslation;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class GroupProductManager extends AbstractProductManager {
    private static final Logger logger = LoggerFactory.getLogger(GroupProductManager.class);
    private final GroupProductDao groupProductDao;
    private final GroupServiceDao groupServiceDao;
    private final TrialDefinitionDao trialDefinitionDao;
    private final UserProductManager userProductManager;
    private final TextsManager textsManager;
    private final FeatureFlags featureFlags;
    private final GroupPromoService groupPromoService;

    public GroupProductManager(
            ProductSetDao productSetDao, ProductLineDao productLineDao, GroupServiceDao groupServiceDao, TextsManager textsManager,
            PromoService promoService, GroupProductDao groupProductDao, TrialDefinitionDao trialDefinitionDao,
            UserProductManager userProductManager, SpringExpressionEvaluator springExpressionEvaluator,
            FeatureFlags featureFlags, GroupPromoService groupPromoService) {
        super(productSetDao, productLineDao, promoService, springExpressionEvaluator);
        this.groupServiceDao = groupServiceDao;
        this.groupProductDao = groupProductDao;
        this.trialDefinitionDao = trialDefinitionDao;
        this.userProductManager = userProductManager;
        this.textsManager = textsManager;
        this.featureFlags = featureFlags;
        this.groupPromoService = groupPromoService;
    }

    public ListF<GroupProduct> findByIds(CollectionF<UUID> productIds) {
        return mapGroupProducts(groupProductDao.findByIds(productIds));
    }

    public GroupProduct findById(UUID productId) {
        return new GroupProduct(groupProductDao.findById(productId), textsManager, userProductManager,
                trialDefinitionDao, groupProductDao);
    }

    public SetF<GroupProduct> getActiveProducts(UUID groupId) {
        SetF<UUID> serviceIds = groupServiceDao.find(groupId, Target.ENABLED)
                .map(GroupService::getGroupProductId)
                .unique();
        return findByIds(serviceIds).unique();
    }

    public SetF<GroupProduct> getActiveProducts(Group group) {
        return getActiveProducts(group.getId());
    }

    @NotNull
    public GroupProduct findProduct(String code) {
        GroupProductEntity product = groupProductDao.findProductByCode(code).orElseThrow(
                () -> new A3ExceptionWithStatus("product_not_found", "Product not found", HttpStatus.SC_404_NOT_FOUND));
        return new GroupProduct(product, textsManager, userProductManager, trialDefinitionDao, groupProductDao);
    }

    public void validateAvailableProduct(PassportUid uid, Group group, GroupProduct groupProduct,
                                         ListF<String> productExperiments) {
        groupProduct.getAvailableTo().ifPresent(to -> {
            if (to.isBeforeNow()) {
                throw new A3ExceptionWithStatus("product_not_available", "Product disabled",
                        HttpStatus.SC_400_BAD_REQUEST);
            }
        });

        if (featureFlags.getCheckAvailableGroupProduct().isEnabled()) {
            ListF<ProductSetEntity> productSets = productSetDao.findByGroupProduct(groupProduct.getId());

            boolean isCorrect = productSets.iterator()
                    .map(ps -> new GroupProductQuery(
                            ps.getKey(),
                            Option.of(uid),
                            Option.of(group),
                            productExperiments,
                            Option.empty()
                    ))
                    .map(this::findGroupProductsWithPromo)
                    .flatMap(l -> l.getGroupProducts().iterator())
                    .find(p -> p.getId().equals(groupProduct.getId()))
                    .isPresent();

            if (!isCorrect) {
                logger.warn("Group product not available: groupProduct {}, productSets: {}", groupProduct, productSets);
                throw new A3ExceptionWithStatus("product_not_allowed", "Product not allowed",
                        HttpStatus.SC_400_BAD_REQUEST);
            }
        }
    }

    public Option<LimitedTimeProductLineB2b> findProductLineWithPromo(GroupProductQuery query) {
        Option<ProductSetEntity> productSetO = productSetDao.findByKey(query.getProductSetKey());
        if (!productSetO.isPresent()) {
            throw new A3ExceptionWithStatus(
                    "unknown_productset", "Product set '" + query.getProductSetKey() + "' not found",
                    HttpStatus.SC_404_NOT_FOUND);
        }
        return selectProductLine(
                productLineDao.findByProductSetId(productSetO.get().getId()),
                false,
                query.getUid(),
                query.getGroup(),
                query.getProductExperiments(),
                query.getPromoTemplateFromPromoCode()
        );
    }

    public LimitedTimeGroupProducts findGroupProductsWithPromo(GroupProductQuery query) {
        return findProductLineWithPromo(query)
                .map(productLine -> new LimitedTimeGroupProducts(
                        getAvailableProducts(query.getGroup(), productLine),
                        productLine.getAvailableUntil(),
                        productLine.getPromo()
                ))
                .orElse(LimitedTimeGroupProducts.EMPTY);
    }

    private ListF<GroupProduct> getAvailableProducts(Option<Group> group, LimitedTimeProductLineB2b productLine) {
        ListF<GroupProduct> products = getProductsByLineId(productLine.getProductLine().getId());
        ListF<GroupProduct> activeProducts = Cf.arrayList();
        activeProducts.addAll(group.flatMap(this::getActiveProducts));
        activeProducts.addAll(group.flatMapO(Group::getParentGroup).flatMap(this::getActiveProducts));
        SetF<GroupProduct> availableAddons = activeProducts
                .filter(product -> product.getProductType() == GroupProductType.MAIN)
                .flatMap(GroupProduct::getAvailableAddons)
                .unique();
        return products.filter(product -> product.getProductType() != GroupProductType.ADDON
                || availableAddons.containsTs(product));
    }


    public Option<LimitedTimeProductLineB2b> selectProductLine(
            ListF<ProductLineEntity> productLines,
            boolean excludePromoLine,
            Option<PassportUid> uid,
            Option<Group> group,
            ListF<String> productExperiments,
            Option<UUID> promoTemplateFromPromoCode
    ) {
        MapF<UUID, AbstractGroupPromoTemplate> productLineWithPromo =
                groupPromoService.readyForUsingGroupPromos(group, promoTemplateFromPromoCode);

        SelectionContext context = new SelectionContext(productExperiments);

        Option<LimitedTimeProductLineB2b> result = productLines
                .filter(line -> !excludePromoLine || !productLineWithPromo.containsKeyTs(line.getId()))
                .sorted(productLineComparator(excludePromoLine, productLineWithPromo))
                .iterator()
                .map(pl -> {
                    Option<AbstractGroupPromoTemplate> promo = productLineWithPromo.getO(pl.getId());
                    ProductLineAvailability available = availableProductLine(
                            uid,
                            group,
                            context,
                            promo,
                            pl,
                            promoTemplateFromPromoCode
                    );
                    return Tuple3.tuple(pl, available, promo);
                })
                .find(t -> t._2.isAvailable())
                .map(t -> new LimitedTimeProductLineB2b(t._1, t._2.getAvailableUntil(), t._3));

        if (result.isPresent()) {
            logger.info("defined product line for group {}: {}", group, result);
        } else {
            logger.info("none of product lines {} is appreciable", productLines);
        }

        return result;
    }

    private Comparator<ProductLineEntity> productLineComparator(boolean excludePromoLine, MapF<UUID, ?> promos) {
        return excludePromoLine
                ? ProductLineEntity.byOrderNum()
                : ProductLineEntity.topWithPromoThenByOrderNum(promos);

    }

    protected ProductLineAvailability testPromoProductLineAvailability(
            AbstractGroupPromoTemplate promoTemplate,
            ProductLineEntity line,
            SelectionContext context,
            Option<PassportUid> uid,
            Option<Group> group,
            Option<UUID> promoTemplateFromPromoCode
    ) {
        ProductLineAvailability lineAvailability = testProductLineAvailability(line, context, uid, group);
        if (!lineAvailability.isAvailable()) {
            return lineAvailability;
        }

        Option<Instant> promoToDateO = promoTemplateFromPromoCode.isSome(promoTemplate.getId())
                ? promoTemplate.getToDate()
                : promoTemplate.canBeUsedUntilDate(group);

        if (!promoToDateO.isPresent()) {
            return lineAvailability;
        }

        Option<Instant> availableUntil = lineAvailability.getAvailableUntil();
        if (!availableUntil.isPresent() || promoToDateO.get().isBefore(availableUntil.get())) {
            return ProductLineAvailability.availableUntil(promoToDateO.get());
        }

        return lineAvailability;
    }

    private ProductLineAvailability availableProductLine(
            Option<PassportUid> uid,
            Option<Group> group,
            SelectionContext context,
            Option<AbstractGroupPromoTemplate> promoTemplate,
            ProductLineEntity pl,
            Option<UUID> promoTemplateFromPromoCode
    ) {
        return promoTemplate
                .map(promo -> testPromoProductLineAvailability(
                                promo,
                                pl,
                                context,
                                uid,
                                group,
                                promoTemplateFromPromoCode
                        )
                )
                .orElseGet(() -> testProductLineAvailability(pl, context, uid, group));
    }


    public ListF<GroupProduct> getProductsByLineId(UUID lineId) {
        ListF<GroupProductEntity> products = groupProductDao.findByProductLine(lineId);
        return mapGroupProducts(products);
    }

    private ListF<GroupProduct> mapGroupProducts(ListF<GroupProductEntity> products) {
        MapF<UUID, UserProduct> userProducts = userProductManager
                .findByIds(products.map(GroupProductEntity::getUserProductId))
                .toMapMappingToKey(UserProduct::getId);
        ListF<UUID> tankerKeyIds = products.filterMap(GroupProductEntity::getTitleTankerKeyId);

        MapF<UUID, TankerTranslation> translationsByTankerKeyId =
                textsManager.findTranslations(tankerKeyIds.stableUnique());

        MapF<UUID, TrialDefinitionEntity> trialDefinitionByIds = trialDefinitionDao.findByIds(
                        products.flatMap(GroupProductEntity::getTrialDefinitionId))
                .toMap(TrialDefinitionEntity::getId, Function.identityF());

        return products.map(p -> new GroupProduct(p,
                p.getTitleTankerKeyId().filterMap(translationsByTankerKeyId::getO),
                userProducts.getOrThrow(p.getUserProductId()),
                p.getTrialDefinitionId().filterMap(trialDefinitionByIds::getO),
                () -> mapGroupProducts(groupProductDao.findAvailableAddons(p.getId())),
                () -> mapGroupProducts(groupProductDao.findEligibleMainProducts(p.getId()))
        ));
    }
}
