package ru.yandex.chemodan.app.psbilling.core.products;

import java.math.BigDecimal;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialDefinitionEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialType;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.Validate;

public class TrialDefinition extends DefaultObject {
    private final TrialDefinitionEntity entity;

    public TrialDefinition(TrialDefinitionEntity entity) {
        this.entity = entity;
    }

    public UUID getId() {
        return entity.getId();
    }

    public CustomPeriod getDurationOrThrow() {
        Validate.equals(entity.getType(), TrialType.CONSTANT_PERIOD);
        return entity.getDuration().getOrThrow("constant period trial type expected");
    }

    public boolean isHidden() {
        return entity.isHidden();
    }

    public TrialType getType() {
        return entity.getType();
    }

    public BigDecimal getPrice() {
        return entity.getPrice();
    }

    public Option<Instant> getEndDate() {
        return entity.getEndDate();
    }

    public Option<CustomPeriod> getDuration() {
        return entity.getDuration();
    }

    public Option<String> getSingleUsageComparisonKey() {
        return entity.getSingleUsageComparisonKey();
    }

    public void ensurePriceZero() {
        Validate.isTrue(BigDecimal.ZERO.compareTo(entity.getPrice()) == 0);
    }

    public Instant calculateEndDate() {
        if (getType() == TrialType.UNTIL_DATE && getEndDate().isPresent()) {
            return getEndDate().get();
        }
        if (getType() == TrialType.CONSTANT_PERIOD && getDuration().isPresent()) {
            CustomPeriod period = getDuration().get();
            Instant now = Instant.now();
            return now.plus(period.toDurationFrom(now));
        }

        throw new IllegalStateException(this.toString());
    }

}
