package ru.yandex.chemodan.app.psbilling.core.products;

import java.util.UUID;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.TrialDefinitionDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductOwnerDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductBucketDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductPeriodDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductPricesDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialDefinitionEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductFeatureEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductOwner;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductPeriodEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductPriceEntity;
import ru.yandex.chemodan.app.psbilling.core.texts.TankerTranslation;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;
import ru.yandex.misc.lang.DefaultObject;

public class UserProduct extends DefaultObject {
    private final UserProductEntity productEntity;
    private final Function<UserProduct, MapF<UserProductPeriod, ListF<UserProductPrice>>> pricesProvider;
    private final Function0<ListF<UserProductFeature>> featuresProvider;
    private final Function0<Option<TankerTranslation>> titleProvider;
    private final Function0<Option<TrialDefinition>> trialDefinitionProvider;
    private final Function0<ProductOwner> productOwnerProvider;
    private final Function0<SetF<UUID>> conflictingProductsProvider;
    private final Function0<SetF<String>> bucketCodesProvider;

    UserProduct(UserProductEntity productEntity, UserProductPricesDao userProductPricesDao,
                UserProductPeriodDao userProductPeriodDao, ProductFeatureDao productFeatureDao,
                TextsManager textsManager,
                TrialDefinitionDao trialDefinitionDao, ProductOwnerDao productOwnerDao,
                UserProductBucketDao userProductBucketDao) {
        this(productEntity,
                (userProduct) -> {
                    ListF<UserProductPeriodEntity> periodsEntity =
                            userProductPeriodDao.findByUserProduct(productEntity.getId());
                    MapF<UUID, ListF<UserProductPriceEntity>> prices =
                            userProductPricesDao.findByPeriodIds(periodsEntity.map(UserProductPeriodEntity::getId));
                    return periodsEntity
                            .map(p -> new UserProductPeriod(p, userProduct, prices.getOrElse(p.getId(), Cf.list())))
                            .toMap(Function.identityF(), UserProductPeriod::getPrices);
                },
                () -> {
                    ListF<ProductFeatureEntity> features =
                            productFeatureDao.findEnabledByUserProduct(productEntity.getId());
                    MapF<UUID, TankerTranslation> translations = textsManager.findTranslations(
                            features.filterMap(ProductFeatureEntity::getDescriptionTankerKeyId)).plus(
                            textsManager.findTranslations(
                                    features.filterMap(ProductFeatureEntity::getGroupTankerKeyId))).plus(
                            textsManager.findTranslations(
                                    features.filterMap(ProductFeatureEntity::getValueTankerKeyId)));
                    return features.map(f -> new UserProductFeature(
                            f,
                            f.getDescriptionTankerKeyId().filterMap(translations::getO),
                            f.getGroupTankerKeyId().filterMap(translations::getO),
                            f.getValueTankerKeyId().filterMap(translations::getO)));
                },
                () -> productEntity.getTitleTankerKeyId().map(textsManager::findTranslation),
                () -> productEntity.getTrialDefinitionId().map(trialDefinitionDao::findById).map(TrialDefinition::new),
                () -> productOwnerDao.findById(productEntity.getProductOwnerId()),
                userProductBucketDao::getUserProductBuckets
        );
    }

    UserProduct(UserProductEntity productEntity, ListF<UserProductPeriodEntity> periods,
                MapF<UUID, ListF<UserProductPriceEntity>> prices,
                ListF<UserProductFeature> features, Option<TankerTranslation> title,
                Option<TrialDefinitionEntity> trialDefinition, ProductOwner productOwner,
                MapF<String, SetF<UUID>> userProductBuckets) {
        this(productEntity, (userProduct) -> periods
                        .map(p -> new UserProductPeriod(p, userProduct, prices.getOrElse(p.getId(), Cf.list())))
                        .toMap(Function.identityF(), UserProductPeriod::getPrices),
                () -> features, () -> title, () -> trialDefinition.map(TrialDefinition::new), () -> productOwner,
                () -> userProductBuckets);
    }

    private UserProduct(UserProductEntity productEntity,
                        Function<UserProduct, MapF<UserProductPeriod, ListF<UserProductPrice>>> pricesProvider,
                        Function0<ListF<UserProductFeature>> featuresProvider,
                        Function0<Option<TankerTranslation>> titleProvider,
                        Function0<Option<TrialDefinition>> trialDefinitionProvider,
                        Function0<ProductOwner> productOwnerProvider,
                        Function0<MapF<String, SetF<UUID>>> allBucketsProvider) {

        Function0<MapF<String, SetF<UUID>>> allBucketsProviderMemoized = allBucketsProvider.memoize();
        Function0<SetF<UUID>> conflictingProductsFunction = () -> allBucketsProviderMemoized.apply().values()
                .filter(b -> b.containsTs(productEntity.getId())).flatMap(Function.identityF())
                .unique();
        Function0<SetF<String>> bucketCodesFunction =
                () -> allBucketsProviderMemoized.apply().filterValues(b -> b.containsTs(productEntity.getId())).keySet();

        this.productEntity = productEntity;
        this.pricesProvider = pricesProvider.memoize();
        this.featuresProvider = featuresProvider.memoize();
        this.titleProvider = titleProvider.memoize();
        this.trialDefinitionProvider = trialDefinitionProvider.memoize();
        this.productOwnerProvider = productOwnerProvider.memoize();
        this.conflictingProductsProvider = conflictingProductsFunction.memoize();
        this.bucketCodesProvider = bucketCodesFunction.memoize();
    }

    public boolean conflictsWith(UUID userProductId) {
        return conflictingProductsProvider.apply().containsTs(userProductId);
    }

    public boolean conflictsWithAny(CollectionF<UUID> userProductIds) {
        SetF<UUID> conflicting = conflictingProductsProvider.apply();
        return userProductIds.stream().anyMatch(conflicting::containsTs);
    }

    public SetF<String> containsInBucketWithCodes() {
        return bucketCodesProvider.apply();
    }

    public boolean isMail360Product() {
        return containsInBucketWithCodes().stream().anyMatch(bucket -> bucket.startsWith(UserProductManager.MAIL_360_BUCKET_PREFIX));
    }

    public SetF<UUID> getConflictingProducts() {
        return conflictingProductsProvider.apply();
    }

    public Option<Integer> getTrustServiceId() {
        return productEntity.getTrustServiceId();
    }

    public Option<TankerTranslation> getTitle() {
        return titleProvider.apply();
    }

    public MapF<UserProductPeriod, ListF<UserProductPrice>> getProductPrices() {
        return pricesProvider.apply(this);
    }

    public ListF<UserProductPeriod> getProductPeriods() {
        return pricesProvider.apply(this).keys();
    }

    public ListF<UserProductFeature> getFeatures() {
        return featuresProvider.apply();
    }

    public Option<TrialDefinition> getTrialDefinition() {
        return trialDefinitionProvider.apply();
    }

    public UUID getId() {
        return productEntity.getId();
    }

    public String getCode() {
        return productEntity.getCode();
    }

    public String getCodeFamily() {
        return productEntity.getCodeFamily();
    }

    public BillingType getBillingType() {
        return productEntity.getBillingType();
    }

    public boolean isBestOffer() {
        return productEntity.isBestOffer();
    }

    public boolean isAllowAutoProlong() {
        return productEntity.isAllowAutoProlong();
    }

    public ProductOwner getProductOwner() {
        return productOwnerProvider.apply();
    }

    public Option<String> getTrustSubsChargingRetryDelay() {
        return productEntity.getTrustSubsChargingRetryDelay();
    }

    public Option<String> getTrustSubsChargingRetryLimit() {
        return productEntity.getTrustSubsChargingRetryLimit();
    }

    public Option<String> getTrustSubsGracePeriod() {
        return productEntity.getTrustSubsGracePeriod();
    }
}
