package ru.yandex.chemodan.app.psbilling.core.products;

import java.math.BigDecimal;
import java.util.NoSuchElementException;
import java.util.UUID;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.TrialDefinitionDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductFeatureDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductLineDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductOwnerDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductSetDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductBucketDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductPeriodDao;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductPricesDao;
import ru.yandex.chemodan.app.psbilling.core.dao.users.UserServiceDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.TrialDefinitionEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.entities.products.FeatureScope;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductFeatureEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductLineEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.ProductOwner;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductPeriodEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductPriceEntity;
import ru.yandex.chemodan.app.psbilling.core.promos.PromoService;
import ru.yandex.chemodan.app.psbilling.core.texts.TankerTranslation;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;

import static ru.yandex.bolts.function.Function.identityF;

public class UserProductManager extends AbstractProductManager {
    private static final int BATCH_SIZE = 500;

    private final TextsManager textsManager;
    private final UserProductDao userProductDao;
    private final UserProductPricesDao userProductPricesDao;
    private final UserProductPeriodDao userProductPeriodDao;
    private final ProductFeatureDao productFeatureDao;
    private final UserServiceDao userServiceDao;
    private final TrialDefinitionDao trialDefinitionDao;
    private final ProductOwnerDao productOwnerDao;
    private final UserProductBucketDao userProductBucketDao;
    public static final String MAIL_360_BUCKET_PREFIX = "mail360";

    public UserProductManager(
            ProductSetDao productSetDao, ProductLineDao productLineDao, UserProductDao userProductDao,
            UserProductPricesDao userProductPricesDao, ProductFeatureDao productFeatureDao, TextsManager textsManager,
            PromoService promoService, UserServiceDao userServiceDao, TrialDefinitionDao trialDefinitionDao,
            UserProductPeriodDao userProductPeriodDao, ProductOwnerDao productOwnerDao,
            UserProductBucketDao userProductBucketDao, SpringExpressionEvaluator springExpressionEvaluator) {
        super(productSetDao, productLineDao, promoService, springExpressionEvaluator);
        this.userProductDao = userProductDao;
        this.userProductPricesDao = userProductPricesDao;
        this.productFeatureDao = productFeatureDao;
        this.userServiceDao = userServiceDao;
        this.trialDefinitionDao = trialDefinitionDao;
        this.userProductPeriodDao = userProductPeriodDao;
        this.productOwnerDao = productOwnerDao;
        this.userProductBucketDao = userProductBucketDao;
        this.textsManager = textsManager;
    }

    public LimitedTimeUserProducts findProductSet(String productSetCode, Option<PassportUid> uidO) {
        return findProductSet(productSetCode, uidO, true);
    }

    public LimitedTimeUserProducts findProductSet(String productSetCode, Option<PassportUid> uidO, boolean includePromoLines) {
        Option<LimitedTimeProductLine> productLineO = findProductLine(productSetCode, uidO, includePromoLines);
        if (!productLineO.isPresent()) {
            return LimitedTimeUserProducts.EMPTY;
        }
        LimitedTimeProductLine limitedTimeProductLine = productLineO.get();

        ListF<UserProductEntity> products = userProductDao
                .findByProductLine(limitedTimeProductLine.getProductLine().getId())
                .filter(p -> !p.getAvailableFrom().isPresent() || p.getAvailableFrom().get().isBeforeNow());
        return new LimitedTimeUserProducts(mapUserProducts(products), limitedTimeProductLine.getAvailableUntil(), limitedTimeProductLine.getPromo());
    }

    public ListF<UserProduct> findByIds(ListF<UUID> userProductIds) {
        if (userProductIds.isEmpty()) {
            return Cf.list();
        }
        if (userProductIds.size() < 2) {
            return Cf.list(findById(userProductIds.single()));
        }
        return mapUserProducts(userProductDao.findByIds(userProductIds));
    }

    public UserProduct findById(UUID userProductId) {
        return new UserProduct(userProductDao.findById(userProductId), userProductPricesDao, userProductPeriodDao,
                productFeatureDao, textsManager, trialDefinitionDao, productOwnerDao, userProductBucketDao);
    }

    public Option<UserProduct> findByCodeO(String userProductCode) {
        Option<UserProductEntity> userProductO = userProductDao.findByCodeO(userProductCode);
        return userProductO.map(p ->
                new UserProduct(p, userProductPricesDao, userProductPeriodDao,
                        productFeatureDao, textsManager, trialDefinitionDao, productOwnerDao, userProductBucketDao));
    }

    public ListF<UserProduct> findByBillingTypes(BillingType... billingTypes) {
        return mapUserProducts(userProductDao.findByBillingTypes(billingTypes));
    }

    public UserProductPrice findPrice(UUID priceId) {
        return new UserProductPrice(
                userProductPricesDao.findById(priceId), userProductPeriodDao, userProductPricesDao, this);
    }

    public ListF<UserProductPrice> findPrices(ListF<UUID> pricesIds) {
        if (pricesIds.isEmpty()) {
            return Cf.list();
        }
        if (pricesIds.size() < 2) {
            return Cf.list(findPrice(pricesIds.single()));
        }
        return mapPrices(userProductPricesDao.findByIds(pricesIds));
    }

    private ListF<UserProductPrice> mapPrices(ListF<UserProductPriceEntity> entities) {
        MapF<UUID, UserProductPeriodEntity> periods = userProductPeriodDao
                .findByIds(entities.map(UserProductPriceEntity::getUserProductPeriodId).stableUnique())
                .toMapMappingToKey(UserProductPeriodEntity::getId);
        return entities.map(e -> new UserProductPrice(e,
                new UserProductPeriod(periods.getOrThrow(e.getUserProductPeriodId()), this, userProductPricesDao)));
    }

    public UserProductPeriod findPeriodByCode(String code) {
        Option<UserProductPeriodEntity> period = userProductPeriodDao.findByCode(code);
        return period.map(p -> new UserProductPeriod(p, this, userProductPricesDao))
                .orElseThrow(() -> new NoSuchElementException("Product period not found by code " + code));
    }

    public Option<UserProductPeriod> findPeriodByCodeO(String code) {
        Option<UserProductPeriodEntity> period = userProductPeriodDao.findByCode(code);
        return period.map(p -> new UserProductPeriod(p, this, userProductPricesDao));
    }

    private ListF<UserProduct> mapUserProducts(ListF<UserProductEntity> products) {
        ListF<UUID> userProductIds = products.map(UserProductEntity::getId);

        MapF<UUID, ListF<ProductFeatureEntity>> featuresByProductIds =
                productFeatureDao.findEnabledByUserProductIds(userProductIds);
        MapF<UUID, ProductOwner> owners = productOwnerDao
                .findByIds(products.map(UserProductEntity::getProductOwnerId).unique().toList())
                .toMapMappingToKey(ProductOwner::getId);

        ListF<UUID> tankerKeyIds = products.filterMap(UserProductEntity::getTitleTankerKeyId).plus(
                featuresByProductIds.values().flatMap(identityF())
                        .filterMap(ProductFeatureEntity::getDescriptionTankerKeyId)).plus(
                featuresByProductIds.values().flatMap(identityF())
                        .filterMap(ProductFeatureEntity::getGroupTankerKeyId)).plus(
                featuresByProductIds.values().flatMap(identityF())
                        .filterMap(ProductFeatureEntity::getValueTankerKeyId));
        MapF<UUID, TankerTranslation> translationsByTankerKeyId =
                textsManager.findTranslations(tankerKeyIds.stableUnique());

        MapF<UUID, ListF<UserProductPeriodEntity>> periodsByUserProductIds =
                userProductPeriodDao.findByUserProductIds(userProductIds);
        MapF<UUID, ListF<UserProductPriceEntity>> pricesByPeriodIds =
                userProductPricesDao.findByPeriodIds(
                        periodsByUserProductIds.values().flatMap(identityF()).map(UserProductPeriodEntity::getId));

        MapF<UUID, TrialDefinitionEntity> trialDefinitionByIds = trialDefinitionDao.findByIds(
                products.flatMap(UserProductEntity::getTrialDefinitionId))
                .toMap(TrialDefinitionEntity::getId, identityF());

        MapF<String, SetF<UUID>> conflictingProductsBuckets = userProductBucketDao.getUserProductBuckets();

        return products.map(p -> new UserProduct(p,
                periodsByUserProductIds.getOrElse(p.getId(), Cf.list()),
                pricesByPeriodIds,
                featuresByProductIds.getO(p.getId()).orElse(Cf.list())
                        .map(f -> new UserProductFeature(
                                f,
                                f.getDescriptionTankerKeyId().filterMap(translationsByTankerKeyId::getO),
                                f.getGroupTankerKeyId().filterMap(translationsByTankerKeyId::getO),
                                f.getValueTankerKeyId().filterMap(translationsByTankerKeyId::getO))
                        ),
                p.getTitleTankerKeyId().filterMap(translationsByTankerKeyId::getO),
                p.getTrialDefinitionId().filterMap(trialDefinitionByIds::getO),
                owners.getOrThrow(p.getProductOwnerId()),
                conflictingProductsBuckets
        ));
    }

    public void addFeatureToProduct(UUID userProductId, UUID featureId, BigDecimal amount,
                                    Option<UUID> descriptionTankerKeyId, String code) {
        productFeatureDao.insert(ProductFeatureDao.InsertData.builder()
                .amount(amount)
                .featureId(Option.of(featureId))
                .userProductId(userProductId)
                .descriptionTankerKeyId(descriptionTankerKeyId)
                .code(code)
                .scope(FeatureScope.USER)
                .build());

        resyncUserProductServices(userProductId);
    }

    public void disableProductFeature(UUID productFeatureId) {
        ProductFeatureEntity productFeature = productFeatureDao.findById(productFeatureId);
        productFeatureDao.disable(productFeature.getId());
        if (!productFeature.getFeatureId().isPresent()) {
            return;
        }

        resyncUserProductServices(productFeature.getUserProductId());
    }

    public void resyncUserProductServices(UUID userProductId) {
        Option<UUID> fromUserServiceId = Option.empty();
        int lastBatchSize = BATCH_SIZE;
        while (lastBatchSize == BATCH_SIZE) {
            ListF<UUID> ids =
                    userServiceDao.findActiveServicesByUserProduct(fromUserServiceId, userProductId, BATCH_SIZE);
            userServiceDao.resetSyncingStatus(ids);
            fromUserServiceId = ids.lastO();
            lastBatchSize = ids.size();
        }

    }

    public void validatePrice(PassportUid uid, UserProductPrice price) {
        if (getUserAvailableProductLinesByProduct(uid, price.getPeriod().getUserProductId(), true).isEmpty()) {
            throw new A3ExceptionWithStatus(
                    "price_unavailable",
                    "the price " + price + " doesn't match any product line available for " + uid,
                    HttpStatus.SC_400_BAD_REQUEST);
        }
    }

    /**
     * Берем продуктовые линейки в которых участвует продукт.
     * Смотрим у каких продуктовых сетах они присутствуют.
     * Выбираем лучшую линейку для каждого продуктового сета.
     *
     * Если в лучшей линейке есть нужный продукт, то это нужная линейка
     *
     */
    public ListF<ProductLineEntity> getUserAvailableProductLinesByProduct(PassportUid uid, UUID productId,
                                                                          boolean includePromoLines) {
        ListF<ProductLineEntity> productLines = productLineDao.findByUserProduct(productId);
        ListF<UUID> productSetIds = productLines.map(ProductLineEntity::getProductSetId).stableUnique();

        MapF<UUID, LimitedTimeProductLine> lineBySet = productLineDao.findByProductSetIds(productSetIds)
                .groupBy(ProductLineEntity::getProductSetId)
                .mapValues(lines -> selectProductLine(lines, Option.of(uid), includePromoLines))
                .filterValues(Option::isPresent)
                .mapValues(Option::get);

        return productLines.filter(productLine ->
                lineBySet.getO(productLine.getProductSetId())
                        .map(LimitedTimeProductLine::getProductLine)
                        .equals(Option.of(productLine)));
    }

    public Option<UserProductFeature> getEnabledFeature(ListF<UserProduct> userProducts,
                                                        Function<UserProductFeature, Boolean> featureSelector) {
        return userProducts.iterator()
                .flatMap(product -> product.getFeatures().iterator())
                .iterator()
                .filter(UserProductFeature::isEnabled)
                .find(featureSelector::apply);
    }
}
