package ru.yandex.chemodan.app.psbilling.core.products;

import java.util.UUID;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.config.Settings;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductPricesDao;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductPeriodEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductPriceEntity;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class UserProductPeriod extends DefaultObject {
    private static final Logger logger = LoggerFactory.getLogger(UserProductPeriod.class);
    private final UserProductPeriodEntity entity;
    private final Function0<UserProduct> productProvider;
    private final Function<UserProductPeriod, ListF<UserProductPrice>> pricesProvider;

    public UserProductPeriod(UserProductPeriodEntity entity, UserProductManager userProductManager,
                             UserProductPricesDao userProductPricesDao) {
        this(entity, () -> userProductManager.findById(entity.getUserProductId()),
                (period) -> userProductPricesDao
                        .findByPeriodId(entity.getId())
                        .map(p -> new UserProductPrice(p, period)));
    }

    UserProductPeriod(UserProductPeriodEntity entity, UserProduct userProduct,
                      ListF<UserProductPriceEntity> prices) {
        this(entity, () -> userProduct, (period) -> prices.map(p -> new UserProductPrice(p, period)));
    }

    UserProductPeriod(UserProductPeriodEntity entity,
                      Function0<UserProduct> productProvider,
                      Function<UserProductPeriod, ListF<UserProductPrice>> pricesProvider) {
        this.entity = entity;
        this.productProvider = productProvider.memoize();
        this.pricesProvider = pricesProvider.memoize();
    }

    public UUID getId() {
        return entity.getId();
    }

    public UserProduct getUserProduct() {
        return productProvider.apply();
    }

    public Option<String> getPackageName() {
        return entity.getPackageName();
    }

    public ListF<UserProductPrice> getPrices() {
        return pricesProvider.apply(this);
    }

    public String getCode() {
        return entity.getCode();
    }

    public CustomPeriod getPeriod() {
        return entity.getPeriod();
    }

    public UUID getUserProductId() {
        return entity.getUserProductId();
    }

    public Option<String> getTrustFiscalTitle() {
        return entity.getTrustFiscalTitle();
    }

    public Option<CustomPeriod> getStartPeriodDuration() {
        return entity.getStartPeriodDuration();
    }

    public Option<Integer> getStartPeriodCount() {
        return entity.getStartPeriodCount();
    }

    public String getTrustProductId(boolean autoprolong) {
        // если на продукте выключено автопродление - то у нас в трасте создается один продукт и его код указывается в
        // базе. если у нас на продукте включено автопродление, то код продукта, который покупается один раз
        // генерится автоматом
        // и не используется нигде, а для покупок используется код, который указан в базе на периоде
        if (!autoprolong) {
            if (getUserProduct().isAllowAutoProlong()) {
                String result = "PS_BILLING_" + getUserProduct().getCode() + "_" +
                        entity.getPeriod().getUnit().value();
                if (entity.getPeriod().getValue() > 1) {
                    result += "_" + entity.getPeriod().getValue();
                }
                result += "_" + getUserProduct().getBillingType().getTrustProductType(false);
                return result;
            }
            return getCode();
        } else {
            return getCode();
        }
    }

    public Option<UserProductPrice> selectPriceForRegion(Settings settings, Option<String> region,
                                                         Option<String> currency) {
        ListF<UserProductPrice> prices =
                getPrices().filter(p -> !settings.getDisabledPriceRegions().containsTs(p.getRegionId()));

        String currencyCode = currency.orElse(UserProductPrice.DEFAULT_CURRENCY);

        if (region.isPresent()) {
            Option<UserProductPrice> price =
                    prices.filter(p -> StringUtils.equalsIgnoreCase(p.getRegionId(), region.get())
                            && StringUtils.equalsIgnoreCase(p.getCurrencyCode(), currencyCode)).firstO();
            if (price.isPresent()) {
                return price;
            }
        }

        Option<UserProductPrice> defaultPrice =
                prices.filter(p -> p.isDefaultRegionPrice()
                        && StringUtils.equalsIgnoreCase(p.getCurrencyCode(), currencyCode)).firstO();
        logger.info("Price for period {} and region {} and currency {} not found, default price {} will be used",
                this, region, currency, defaultPrice);

        return defaultPrice;
    }

    public boolean hasStartPeriod() {
        if (getStartPeriodDuration().isEmpty() && getStartPeriodCount().isEmpty()) {
            return false;
        }
        if (getStartPeriodDuration().isEmpty() != getStartPeriodCount().isEmpty()) {
            logger.error("startPeriodCount and startPeriodDuration should be provided simultaneously");
            return false;
        }
        if (getPrices().stream().anyMatch(price -> price.getStartPeriodPrice().isEmpty())) {
            logger.error("no startPeriod price for period {} with start period ", getCode());
            return false;
        }
        return true;

    }
}
