package ru.yandex.chemodan.app.psbilling.core.promocodes.impl;

import java.security.SecureRandom;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.UUID;

import com.google.common.primitives.Longs;
import lombok.AllArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Instant;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.dao.products.UserProductPricesDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promocodes.GroupPromoCodeActivationDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promocodes.PromoCodeDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promocodes.PromoCodeTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promocodes.UserPromoCodeDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.PromoTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.products.UserProductPriceEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.GroupPromoCodeActivationEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeType;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.UserPromoCodeEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.model.RequestInfo;
import ru.yandex.chemodan.app.psbilling.core.promocodes.PromoCodeService;
import ru.yandex.chemodan.app.psbilling.core.promocodes.activator.PromoCodeActivatorFactory;
import ru.yandex.chemodan.app.psbilling.core.promocodes.exception.PromoCodeCantBeActivatedException;
import ru.yandex.chemodan.app.psbilling.core.promocodes.generator.PromoCodeGenerator;
import ru.yandex.chemodan.app.psbilling.core.promocodes.length.PromoCodeLengthService;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeActivationResult;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeData;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.SafePromoCode;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.activator.AbstractPromoCodeActivator;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.activator.PromoCodeDataActivator;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.error.CheckerRulePromoCodeActivationFail;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.error.UsedPromoCodeActivationFail;
import ru.yandex.chemodan.app.psbilling.core.promocodes.rule.PromoCodeRuleChecker;
import ru.yandex.chemodan.app.psbilling.core.promocodes.rule.PromoCodeRuleCheckerEvaluate;
import ru.yandex.chemodan.app.psbilling.core.promocodes.rule.PromoCodeRuleCheckerResult;
import ru.yandex.chemodan.app.psbilling.core.promocodes.rule.PromoCodeRuleContext;
import ru.yandex.chemodan.app.psbilling.core.promocodes.tasks.GeneratePromoCodeTask;
import ru.yandex.chemodan.app.psbilling.core.promocodes.validation.PromoCodeValidator;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;

@Slf4j
@AllArgsConstructor
public class PromoCodeServiceImpl implements PromoCodeService {
    private final PromoCodeDao promoCodeDao;
    private final UserPromoCodeDao userPromoCodeDao;
    private final GroupPromoCodeActivationDao groupPromoCodeActivationDao;
    private final PromoTemplateDao promoTemplateDao;
    private final UserProductPricesDao productPricesDao;
    private final PromoCodeActivatorFactory activatorFactory;
    private final TransactionTemplate transactionTemplate;
    private final BazingaTaskManager bazingaTaskManager;
    private final PromoCodeLengthService promoCodeLengthService;
    private final PromoCodeRuleCheckerEvaluate promoCodeRuleCheckerEvaluate;
    private final PromoCodeValidator basePromoCodeValidation;
    private final PromoCodeTemplateDao promoCodeTemplateDao;

    public static final char[] ALPHABET = new char[]{
            '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',
            'J', 'K', 'L', 'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z'
    };

    protected static final DynamicProperty<Integer> NUM_ATTEMPTS_TO_GENERATE_CODE = new DynamicProperty<>(
            PromoCodeServiceImpl.class.getName() + ".NUM_ATTEMPTS_TO_GENERATE_CODE", 10);
    private static final DynamicProperty<Integer> BATCH_SIZE =
            new DynamicProperty<>(PromoCodeServiceImpl.class.getName() + ".BATCH_SIZE", 1000);


    private PromoCodeActivationResult commonActivatePromoCode(
            Option<PromoCodeEntity> promoCodeEntityO,
            PassportUid uid,
            Option<Group> groupO
    ) {
        log.info("Activating promo code {} for group {} by uid {}", promoCodeEntityO, uid, groupO);

        PromoCodeData promoCodeData = promoCodeEntityO
                .map(p -> PromoCodeData.byEntityWithLazyTemplate(p, promoCodeTemplateDao::findByIdO))
                .orElseThrow(PromoCodeCantBeActivatedException::new);

        basePromoCodeValidation.validate(promoCodeData);

        PromoCodeDataActivator activator = activatorFactory.buildActivator(promoCodeData, uid, groupO);

        if (!activator.canBeActivated()) {
            log.warn("Promo code can not be activated {}", activator);
            throw new PromoCodeCantBeActivatedException();
        }

        PromoCodeRuleChecker ruleChecker = promoCodeRuleCheckerEvaluate.evaluateRuleChecker(promoCodeData);
        PromoCodeRuleCheckerResult checkerResult = ruleChecker.check(
                promoCodeData,
                Option.of(uid),
                groupO,
                new PromoCodeRuleContext());

        if (!checkerResult.isSuccess()) {
            PromoCodeRuleCheckerResult.Error error = checkerResult.getError()
                    .orElseThrow(() -> new IllegalStateException("CheckerRuleResult is not success and error empty"));

            log.warn("Promo code can not be activated {}. Rule checker {} return error {}", promoCodeData,
                    ruleChecker, error);
            return PromoCodeActivationResult.fail(new CheckerRulePromoCodeActivationFail(error));
        }

        log.debug("Promo code {} can be activated. Starting a transaction to activate {}", promoCodeData, activator);

        PromoCodeData updated = transactionTemplate.execute(status -> activatePromoCode(activator));

        log.info("Finished activating promo code {}, used for uid: {} and group: {}", updated.getCode(), uid, groupO);

        return PromoCodeActivationResult.activated(activator);
    }

    @Override
    public PromoCodeActivationResult activateGroupPromoCode(
            SafePromoCode promoCode,
            Group group,
            PassportUid uid,
            RequestInfo requestInfo
    ) {
        log.info("Activating promo code {} for group {} by uid {} requestInfo {}", promoCode, group, uid, requestInfo);

        Option<GroupPromoCodeActivationEntity> usedPromo =
                groupPromoCodeActivationDao.findByGroupIdAndPromoCode(promoCode, group.getId());

        if (usedPromo.isPresent()) {
            log.warn("Promo code {} already used {} for uid {} group {}", promoCode, usedPromo, uid, group);
            return PromoCodeActivationResult.fail(new UsedPromoCodeActivationFail());
        }

        Option<PromoCodeEntity> promoCodeEntityO = promoCodeDao.findByIdAndTypeO(promoCode, PromoCodeType.B2B);

        return commonActivatePromoCode(promoCodeEntityO, uid, Option.of(group));
    }

    @Override
    public PromoCodeActivationResult activatePromoCode(SafePromoCode promoCode, PassportUid uid,
                                                       RequestInfo requestInfo) {
        log.info("Activating promo code {} for uid {} requestInfo {}", promoCode, uid, requestInfo);

        Option<UserPromoCodeEntity> usedPromo = userPromoCodeDao.findByCodeAndUid(promoCode, uid);

        if (usedPromo.isPresent()) {
            log.warn("Promo code {} already used {} for uid {}", promoCode, usedPromo, uid);
            return PromoCodeActivationResult.fail(new UsedPromoCodeActivationFail());
        }

        Option<PromoCodeEntity> promoCodeEntityO = promoCodeDao.findByIdAndTypeO(promoCode, PromoCodeType.B2C);

        return commonActivatePromoCode(promoCodeEntityO, uid, Option.empty());
    }

    @Override
    public PromoCodeData canActivateGroupPromoCode(
            SafePromoCode promoCode,
            Option<Group> group,
            Option<PassportUid> uid,
            RequestInfo requestInfo
    ) {
        log.info("Can activating promo code {} by uid {} and group {} requestInfo {}", promoCode, uid, group,
                requestInfo);

        PromoCodeData promoCodeData = promoCodeDao.findByIdAndTypeO(promoCode, PromoCodeType.B2B)
                .map(p -> PromoCodeData.byEntityWithLazyTemplate(p, promoCodeTemplateDao::findByIdO))
                .orElseThrow(PromoCodeCantBeActivatedException::new);

        basePromoCodeValidation.validate(promoCodeData);

        if (uid.isPresent() && group.isPresent()) {
            AbstractPromoCodeActivator activatorPromoCodeData =
                    activatorFactory.buildActivator(promoCodeData, uid.get(), group);
            if (activatorPromoCodeData.canBeActivated()) {
                log.warn("Promo code can not be activated {}", activatorPromoCodeData);
                throw new PromoCodeCantBeActivatedException();
            }

            PromoCodeRuleChecker ruleChecker = promoCodeRuleCheckerEvaluate.evaluateRuleChecker(promoCodeData);
            PromoCodeRuleCheckerResult checkerResult = ruleChecker.check(
                    promoCodeData,
                    uid,
                    group,
                    new PromoCodeRuleContext());

            if (!checkerResult.isSuccess()) {
                log.warn("Promo code can not be activated {} for rule checker {} error {}",
                        activatorPromoCodeData, ruleChecker, checkerResult);
                throw new PromoCodeCantBeActivatedException();
            }
        }

        return promoCodeData;
    }

    private PromoCodeData activatePromoCode(PromoCodeDataActivator activator) {
        PromoCodeEntity promoCodeEntity = promoCodeDao.findById(activator.getPromoCodeData().getCode());

        Option<PromoCodeTemplateEntity> template = promoCodeEntity.getTemplateCode()
                .flatMapO(promoCodeTemplateDao::findByIdO);

        PromoCodeData newPromoCodeData = PromoCodeData.byEntityTemplateO(promoCodeEntity, template);

        basePromoCodeValidation.validate(newPromoCodeData);

        activator.activate();

        if (newPromoCodeData.getRemainingActivations().isPresent()) {
            log.info("Promo code {} has non-blank remaining number of activations. Decrementing the number",
                    newPromoCodeData);
            try {
                promoCodeEntity = promoCodeDao.decrementRemainingActivations(newPromoCodeData.getCode());
            } catch (DataIntegrityViolationException e) {
                log.warn("Promo code {} activation error", promoCodeEntity.getCode(), e);
                throw new PromoCodeCantBeActivatedException();
            }
        }
        return PromoCodeData.byEntityTemplateO(promoCodeEntity, template);
    }


    @Override
    public void blockPromoCode(SafePromoCode promoCode, String reason) {
        log.info("Promo code {} will be blocked with reason {}", promoCode, reason);
        if (StringUtils.isBlank(reason)) {
            throw new BadRequestException("Reason must be filled");
        }
        promoCodeDao.blockCode(promoCode, reason);
    }

    private SafePromoCode generateRandomCode(String prefix, long length, SecureRandom random) {
        String code = random.ints(0, ALPHABET.length)
                .limit(length)
                .map(i -> ALPHABET[i])
                .collect(
                        StringBuilder::new,
                        StringBuilder::appendCodePoint,
                        StringBuilder::append
                )
                .toString();

        return SafePromoCode.cons(prefix + code);
    }

    @SneakyThrows
    private PromoCodeGenerator generateCodesInBatches(
            int numToGenerate,
            String prefix,
            PromoCodeDao.InsertData.InsertDataBuilder insertDataBuilder
    ) {
        final long postfixLength = promoCodeLengthService.calcLength(prefix, numToGenerate, ALPHABET.length);
        final SecureRandom secureRandom = new SecureRandom(Longs.toByteArray(System.currentTimeMillis()));

        return new PromoCodeGenerator() {
            private int remainsToGenerate = numToGenerate;
            private int watchDog = NUM_ATTEMPTS_TO_GENERATE_CODE.get();


            @Override
            public boolean hasNext() {
                return remainsToGenerate > 0;
            }

            @Override
            public ListF<SafePromoCode> next() {
                if (!hasNext()) {
                    throw new NoSuchElementException();
                }

                int curBatchSize = Math.min(BATCH_SIZE.get(), remainsToGenerate);
                log.info("Generating batch of size {} with prefix {}. remains to generate {}", curBatchSize, prefix,
                        remainsToGenerate);

                SetF<SafePromoCode> curBatch = Cf.hashSet();
                for (int i = 0; i < curBatchSize; i++) {
                    curBatch.add(generateRandomCode(prefix, postfixLength, secureRandom));
                }
                CollectionF<SafePromoCode> conflicts = promoCodeDao.getConflictingCodes(curBatch);
                log.debug("number of conflicts: {}", conflicts.size());

                curBatch.removeAllTs(conflicts);
                ListF<SafePromoCode> codes = curBatch.toList();

                //all codes are occupied
                if (codes.size() == 0) {
                    watchDog--;
                    if (watchDog == 0) {
                        throw new IllegalStateException("Failed to generate promo codes in " + NUM_ATTEMPTS_TO_GENERATE_CODE.get() + " consecutive attempts ");
                    }
                    log.warn("All codes are occupied. retry attemps remaining: {}", watchDog);
                    return Cf.list();
                } else {
                    watchDog = NUM_ATTEMPTS_TO_GENERATE_CODE.get();
                }

                insertDataBuilder.codes(codes);
                promoCodeDao.create(insertDataBuilder.build());

                remainsToGenerate -= codes.size();

                return codes;
            }
        };
    }

    private UUID findProductPriceByPeriodCode(String periodCode) {
        return productPricesDao.findPricesByPeriodCode(periodCode).firstO()
                .map(UserProductPriceEntity::getId)
                .orElseThrow(
                        () -> new IllegalArgumentException("Can not find product price by period code " + periodCode)
                );
    }

    private UUID findPromoTemplateIdByCode(String promoTemplateCode) {
        return promoTemplateDao.findByCode(promoTemplateCode).map(PromoTemplateEntity::getId)
                .orElseThrow(() -> new IllegalArgumentException("Can not find promo template by id " + promoTemplateCode));
    }

    @Override
    @SneakyThrows
    public PromoCodeGenerator buildPromoCodeGenerator(GeneratePromoCodeTask.Parameters params) {
        log.info("Started generating promo codes with parameters {}", params);

        validate(params);

        PromoCodeDao.InsertData.InsertDataBuilder builder = PromoCodeDao.InsertData.builder()
                .promoCodeStatus(PromoCodeStatus.ACTIVE)
                .promoCodeType(params.getType())
                .numActivations(params.getNumActivations())
                .remainingActivations(params.getNumActivations())
                .toDate(params.getActiveToO())
                .fromDate(params.getActiveFromO().orElse(Instant.now()))
                .productPriceId(params.getProductPeriodCodeO().map(this::findProductPriceByPeriodCode))
                .promoTemplateId(params.getPromoTemplateCodeO().map(this::findPromoTemplateIdByCode));

        Set<String> codes = params.getCodes();
        PromoCodeGenerator generator;
        if (codes.isEmpty()) {
            String prefix = params.getPrefixO().orElse("");
            generator = generateCodesInBatches(params.getNumToGenerate(), prefix, builder);
        } else {
            generator = new PromoCodeGenerator() {

                private boolean hasNext = true;

                @Override
                public boolean hasNext() {
                    return hasNext;
                }

                @Override
                public ListF<SafePromoCode> next() {
                    if (!hasNext()) {
                        throw new NoSuchElementException();
                    }

                    ListF<SafePromoCode> uniqueCodes = Cf.x(codes).map(SafePromoCode::cons);
                    CollectionF<SafePromoCode> conflictingCodes = promoCodeDao.getConflictingCodes(uniqueCodes);

                    if (conflictingCodes.isNotEmpty()) {
                        throw new IllegalArgumentException("Promo codes exist. Codes=" + conflictingCodes);
                    }

                    promoCodeDao.create(builder.codes(uniqueCodes).build());
                    hasNext = false;

                    return uniqueCodes;
                }
            };
        }

        return generator;
    }

    private void validate(GeneratePromoCodeTask.Parameters params) {
        switch (params.getType()) {
            case B2C:
                if (params.getProductPeriodCodeO().isPresent() == params.getPromoTemplateCodeO().isPresent()) {
                    throw new IllegalArgumentException("Can not request " + PromoCodeType.B2C + " promo code for both" +
                            " promo and product period or neither of them");
                }
                break;
            case B2B:
                if (params.getProductPeriodCodeO().isPresent() || params.getPromoTemplateCodeO().isEmpty()) {
                    throw new IllegalArgumentException("Can not request " + PromoCodeType.B2B + " promo code. Enable " +
                            "only promo");
                }
                break;
            default:
                throw new IllegalArgumentException("unable to define promo code type " + params.getType());
        }
    }

    public void scheduleGenerateAndPostPromoCodes(GeneratePromoCodeTask.Parameters parameters) {
        bazingaTaskManager.schedule(new GeneratePromoCodeTask(parameters));
    }
}
