package ru.yandex.chemodan.app.psbilling.core.promocodes.length;

import java.util.concurrent.TimeUnit;

import lombok.RequiredArgsConstructor;

import ru.yandex.chemodan.app.psbilling.core.dao.promocodes.PromoCodeDao;
import ru.yandex.commune.dynproperties.DynamicProperty;

@RequiredArgsConstructor
public class SmartPromoCodeLengthService implements PromoCodeLengthService {
    private final DynamicProperty<Double> ACCEPTABLE_GUESS_PROBABILITY =
            new DynamicProperty<>(SmartPromoCodeLengthService.class.getName() + ".ACCEPTABLE_GUESS_PROBABILITY", 0.1);
    private final DynamicProperty<Double> EXPECTED_BRUTEFORCE_RPS =
            new DynamicProperty<>(SmartPromoCodeLengthService.class.getName() + ".EXPECTED_BRUTEFORCE_RPS", 100.0);
    private final DynamicProperty<Integer> MIN_REQUIRED_POSTFIX_LENGTH =
            new DynamicProperty<>(SmartPromoCodeLengthService.class.getName() + ".MIN_REQUIRED_POSTFIX_LENGTH", 10);
    private final DynamicProperty<Integer> NUM_ATTEMPT_TO_CALC =
            new DynamicProperty<>(SmartPromoCodeLengthService.class.getName() + ".NUM_ATTEMPT_TO_CALC", 100);

    private final PromoCodeDao promoCodeDao;

    /**
     * При генерации пачки промокодов постфикс должен обеспечить защиту от перебора. Ожидаемый срок подбора второго
     * активного промокода должен быть порядка года
     * требование
     * <a href="https://wiki.yandex-team.ru/disk/billing360/projects/promokody/">https://wiki.yandex-team.ru/disk/billing360/projects/promokody/</a>
     * 100 rps * 3600*24*365 = 3153600000 = N - number of attempts possible in 1 year
     * p - probability that user guesses right
     * (1-p)^N > p0 = probability that promocode will not be successfully guessed
     * 1 - p = > p0^(1/N)
     * p = num_codes/num_options < 1 - p0^(1/N)
     * num_options > num_codes/(1-p0^(1/N)) = num_codes/(1-exp(ln(p0)/N)) ~ -N*num_codes/ln(p0)
     **/
    @Override
    public long calcLength(String prefix, long numToGenerate, int alphabetLength) {
        double minAvailableCombinations = minAvailableCombinationsRequired(numToGenerate);
        int minPossibleLength = minPossiblePostfixLength(minAvailableCombinations, alphabetLength);

        //check that there's enough vacant options among already generated promoCodes to generate into
        Integer numAttemptToCalc = NUM_ATTEMPT_TO_CALC.get();
        for (long result = minPossibleLength; result < numAttemptToCalc; result++) {
            long numOccupied = promoCodeDao.calculateNumOccupied(prefix, result);
            double numPossible = Math.pow(alphabetLength, result);
            if (numPossible - numOccupied >= minAvailableCombinations) {
                return result;
            }
        }
        throw new IllegalStateException("Failed to choose minimum possible length in " + numAttemptToCalc +
                " attempts for prefix " + prefix + ", num to generate " + numToGenerate + " and alphabetLength " +
                alphabetLength);
    }

    private double minAvailableCombinationsRequired(long numToGenerate) {
        return Math.ceil(numToGenerate * minAvailableCombinationsMultiplier());
    }

    private int minPossiblePostfixLength(double minAvailableCombinations, int alphabetLength) {
        double minPossible = Math.ceil(Math.log(minAvailableCombinations) / Math.log(alphabetLength));

        return (int) Math.max(MIN_REQUIRED_POSTFIX_LENGTH.get(), minPossible);
    }

    private double minAvailableCombinationsMultiplier() {
        return -EXPECTED_BRUTEFORCE_RPS.get() * (double) TimeUnit.DAYS.toSeconds(365) / Math.log(ACCEPTABLE_GUESS_PROBABILITY.get());
    }
}
