package ru.yandex.chemodan.app.psbilling.core.promocodes.model;

import java.util.Objects;
import java.util.UUID;

import org.apache.commons.lang3.StringUtils;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeStatus;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeType;


public class PromoCodeData {
    private final Function0<PromoCodeEntity> promoCodeEntityProvider;
    private final Function0<Option<PromoCodeTemplateEntity>> promoCodeTemplateEntityProvider;

    private PromoCodeData(
            Function0<PromoCodeEntity> promoCodeEntityProvider,
            Function0<Option<PromoCodeTemplateEntity>> promoCodeTemplateEntityProvider
    ) {
        this.promoCodeEntityProvider = promoCodeEntityProvider.memoize();
        this.promoCodeTemplateEntityProvider = promoCodeTemplateEntityProvider.memoize();
    }

    public static PromoCodeData byEntity(
            PromoCodeEntity codeEntity,
            PromoCodeTemplateEntity templateEntity
    ) {
        return new PromoCodeData(
                () -> codeEntity,
                () -> Option.of(templateEntity)
        );
    }

    public static PromoCodeData byEntityTemplateO(
            PromoCodeEntity codeEntity,
            Option<PromoCodeTemplateEntity> templateEntity
    ) {
        return new PromoCodeData(
                () -> codeEntity,
                () -> templateEntity
        );
    }

    public static PromoCodeData byEntityWithLazyTemplate(
            PromoCodeEntity codeEntity,
            Function<String, Option<PromoCodeTemplateEntity>> templateEntityProvider
    ) {
        return new PromoCodeData(
                () -> codeEntity,
                () -> codeEntity.getTemplateCode()
                        .flatMapO(templateEntityProvider)
        );
    }


    public SafePromoCode getCode() {
        return promoCodeEntityProvider.apply().getCode();
    }


    public Option<Integer> getNumActivation() {
        return promoCodeTemplateEntityProvider.apply()
                .flatMapO(PromoCodeTemplateEntity::getNumActivations)
                .orElse(() -> promoCodeEntityProvider.apply().getNumActivations());
    }


    public Option<Integer> getRemainingActivations() {
        return promoCodeEntityProvider.apply().getRemainingActivations();
    }


    public PromoCodeStatus getStatus() {
        return promoCodeEntityProvider.apply().getStatus();
    }


    public Instant getStatusUpdatedAt() {
        return promoCodeEntityProvider.apply().getStatusUpdatedAt();
    }


    public Option<String> getStatusReason() {
        return promoCodeEntityProvider.apply().getStatusReason();
    }


    public Option<UUID> getPromoTemplateId() {
        return promoCodeTemplateEntityProvider.apply()
                .flatMapO(PromoCodeTemplateEntity::getPromoTemplateId)
                .orElse(() -> promoCodeEntityProvider.apply().getPromoTemplateId());
    }


    public Option<UUID> getUserProductPriceId() {
        return promoCodeTemplateEntityProvider.apply()
                .flatMapO(PromoCodeTemplateEntity::getUserProductPriceId)
                .orElse(() -> promoCodeEntityProvider.apply().getUserProductPriceId());
    }


    public Instant getFromDate() {
        return promoCodeTemplateEntityProvider.apply()
                .map(PromoCodeTemplateEntity::getFromDate)
                .orElseGet(() -> promoCodeEntityProvider.apply().getFromDate());
    }


    public Option<Instant> getToDate() {
        return promoCodeTemplateEntityProvider.apply()
                .flatMapO(PromoCodeTemplateEntity::getToDate)
                .orElse(() -> promoCodeEntityProvider.apply().getToDate());
    }


    public PromoCodeType getType() {
        return promoCodeTemplateEntityProvider.apply()
                .map(PromoCodeTemplateEntity::getType)
                .orElseGet(() -> promoCodeEntityProvider.apply().getType());
    }

    // На период переезда делаем по дефолту строку как EMPTY - PromoCodeRuleEvaluate
    // Далее Template будет обязательным полем для промо кода

    public String getRuleCheckerBeanEl() {
        return promoCodeTemplateEntityProvider.apply()
                .map(PromoCodeTemplateEntity::getRuleCheckerBeanEl)
                .orElseGet(() -> StringUtils.EMPTY);
    }


    @Override
    public String toString() {
        return "PromoCodeData{" +
                "promoCode=" + getCode() +
                '}';
    }


    public String fullToString() {
        return "PromoCodeData{" +
                "promoCodeEntity=" + promoCodeEntityProvider.apply() +
                "promoCodeTemplateEntity=" + promoCodeTemplateEntityProvider.apply() +
                '}';
    }


    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PromoCodeData that = (PromoCodeData) o;
        return Objects.equals(getCode(), that.getCode());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getCode());
    }
}
