package ru.yandex.chemodan.app.psbilling.core.promocodes.model.activator;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.dao.promocodes.GroupPromoCodeActivationDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationArea;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.group.GroupPromoEntity;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeData;
import ru.yandex.chemodan.app.psbilling.core.promos.groups.AbstractGroupPromoTemplate;
import ru.yandex.chemodan.app.psbilling.core.promos.v2.GroupPromoService;
import ru.yandex.inside.passport.PassportUid;

@Slf4j
public class GroupPromoTemplatePromoCodeActivator extends AbstractPromoCodeActivator {
    private final GroupPromoService groupPromoService;
    private final GroupPromoCodeActivationDao groupPromoCodeActivationDao;
    private final Function0<AbstractGroupPromoTemplate> promoTemplateSupplier;
    private final Group group;
    private final PassportUid uid;

    public GroupPromoTemplatePromoCodeActivator(
            PromoCodeData promoCodeData,
            Group group,
            PassportUid uid,
            GroupPromoService groupPromoService,
            GroupPromoCodeActivationDao groupPromoCodeActivationDao
    ) {
        super(promoCodeData);
        this.groupPromoService = groupPromoService;
        this.groupPromoCodeActivationDao = groupPromoCodeActivationDao;

        this.group = group;
        this.uid = uid;
        this.promoTemplateSupplier = getPromoTemplate().memoize();
    }

    private Function0<AbstractGroupPromoTemplate> getPromoTemplate() {
        return () -> groupPromoService.findById(getPromoCodeData().getPromoTemplateId().get());
    }

    @Override
    public boolean canBeActivated() {
        AbstractGroupPromoTemplate promoTemplate = promoTemplateSupplier.apply();

        if (PromoApplicationArea.PER_GROUP != promoTemplate.getApplicationArea()) {
            log.error("Only per group promos can be activated via promo codes. promo {} promo code {}", promoTemplate,
                    getPromoCodeData().getCode()
            );
            throw new IllegalStateException("Illegal promo configuration. Only per-group promos can be activated via " +
                    "promo codes. promo id is " + promoTemplate.getCode());
        }

        if (!promoTemplate.isActive() || !promoTemplate.isAvailableFor(group, true)) {
            log.warn("Promo {} cant be activated for group {}. promo code {}", promoTemplate, group, getPromoCodeData().getCode());
            return false;
        }

        log.debug("Promo code {} is ok to proceed based on promo application type and dates {}", getPromoCodeData().getCode(),
                promoTemplate
        );
        return true;
    }

    @Override
    protected void activateImpl() {
        AbstractGroupPromoTemplate promoTemplate = promoTemplateSupplier.apply();
        Option<GroupPromoEntity> groupPromoEntityO = promoTemplate.activatePromo(group, true, null);

        GroupPromoEntity entity = groupPromoEntityO
                .orElseThrow(() -> {
                    log.error("Activation promo failed. promoCode {}, promoTemplate {}, group {}, uid {}",
                            getPromoCodeData().getCode(),
                            promoTemplate,
                            group,
                            uid);
                    return new IllegalStateException("Activation promo failed");
                });

        groupPromoCodeActivationDao.create(
                GroupPromoCodeActivationDao.InsertData.builder()
                        .code(getPromoCodeData().getCode())
                        .uid(uid)
                        .groupId(group.getId())
                        .groupPromoId(entity.getId())
                        .build()
        );
    }

    @Override
    public String logUsedFor() {
        return group.toString();
    }

    @Override
    public String getPromoCodeActivationType() {
        return "promo_template";
    }
}
