package ru.yandex.chemodan.app.psbilling.core.promocodes.model.activator;

import java.util.UUID;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.billing.users.UserBillingService;
import ru.yandex.chemodan.app.psbilling.core.dao.promocodes.UserPromoCodeDao;
import ru.yandex.chemodan.app.psbilling.core.entities.products.BillingType;
import ru.yandex.chemodan.app.psbilling.core.products.UserProduct;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductManager;
import ru.yandex.chemodan.app.psbilling.core.products.UserProductPeriod;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeData;
import ru.yandex.chemodan.app.psbilling.core.users.UserService;
import ru.yandex.chemodan.app.psbilling.core.users.UserServiceManager;
import ru.yandex.inside.passport.PassportUid;

@Slf4j
public class UserProductPromoCodeActivator extends AbstractPromoCodeActivator {
    private final UserBillingService userBillingService;
    private final UserServiceManager userServiceManager;
    private final PassportUid uid;
    private final UserPromoCodeDao userPromoCodeDao;
    private final Function0<UserProductPeriod> userProductPeriodSupplier;

    public UserProductPromoCodeActivator(
            PromoCodeData promoCodeData,
            PassportUid uid,
            UserProductManager userProductManager,
            UserServiceManager userServiceManager,
            UserBillingService userBillingService,
            UserPromoCodeDao userPromoCodeDao
    ) {
        super(promoCodeData);
        this.userServiceManager = userServiceManager;
        this.userBillingService = userBillingService;
        this.userPromoCodeDao = userPromoCodeDao;
        this.uid = uid;
        this.userProductPeriodSupplier = getUserProductPeriodFunction0(userProductManager).memoize();
    }

    private Function0<UserProductPeriod> getUserProductPeriodFunction0(UserProductManager userProductManager) {
        return () -> userProductManager
                .findPrice(getPromoCodeData().getUserProductPriceId().get())
                .getPeriod();
    }

    @Override
    public boolean canBeActivated() {
        UserProductPeriod userProductPeriod = userProductPeriodSupplier.apply();
        UserProduct promoProduct = userProductPeriod.getUserProduct();

        if (!BillingType.FREE.equals(promoProduct.getBillingType())) {
            log.warn("billing type is not free for product {}. can not activate promo code {}", promoProduct,
                    getPromoCodeData().getCode());
            return false;
        }

        ListF<UserService> services = userServiceManager.findEnabled(uid.toString(), Option.empty());
        ListF<UUID> enabledProducts = services.map(UserService::getUserProductId);

        boolean conflicts = promoProduct.conflictsWithAny(enabledProducts);
        if (conflicts) {
            log.warn("Promo product {} conflicts with one or more of products enabled for user {}. can not activate " +
                    "promo code {}", promoProduct.getCode(), enabledProducts, getPromoCodeData().getCode());
            return false;
        }
        log.debug("Promo product {} is ok to proceed for promo code {}", promoProduct.getCode(), getPromoCodeData().getCode());
        return true;
    }

    @Override
    protected void activateImpl() {
        UserProductPeriod userProductPeriod = userProductPeriodSupplier.apply();
        UUID orderId = userBillingService.createFreeOrder(uid, userProductPeriod.getCode());

        userPromoCodeDao.create(
                UserPromoCodeDao.InsertData.builder()
                        .code(getPromoCodeData().getCode())
                        .uid(uid)
                        .orderId(Option.of(orderId))
                        .build()
        );
    }

    @Override
    public String logUsedFor() {
        return uid.toString();
    }

    @Override
    public String getPromoCodeActivationType() {
        return "product";
    }
}
