package ru.yandex.chemodan.app.psbilling.core.promocodes.model.activator;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.dao.promocodes.UserPromoCodeDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.UserPromoDao;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationArea;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.UserPromoEntity;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeData;
import ru.yandex.chemodan.app.psbilling.core.promos.PromoService;
import ru.yandex.chemodan.app.psbilling.core.promos.PromoTemplate;
import ru.yandex.inside.passport.PassportUid;

@Slf4j
public class UserPromoTemplatePromoCodeActivator extends AbstractPromoCodeActivator {
    private final UserPromoDao userPromoDao;
    private final PromoService promoService;
    private final UserPromoCodeDao userPromoCodeDao;

    private final Function0<PromoTemplate> promoTemplateSupplier;
    private final PassportUid uid;

    public UserPromoTemplatePromoCodeActivator(
            PromoCodeData promoCodeData,
            PassportUid uid,
            UserPromoDao userPromoDao,
            PromoService promoService,
            UserPromoCodeDao userPromoCodeDao
    ) {
        super(promoCodeData);
        this.userPromoDao = userPromoDao;
        this.promoService = promoService;
        this.userPromoCodeDao = userPromoCodeDao;

        this.uid = uid;
        this.promoTemplateSupplier = getPromoTemplate().memoize();
    }

    private Function0<PromoTemplate> getPromoTemplate() {
        return () -> promoService.findById(getPromoCodeData().getPromoTemplateId().get());
    }

    @Override
    public boolean canBeActivated() {
        PromoTemplate promoTemplate = promoTemplateSupplier.apply();

        if (PromoApplicationArea.PER_USER != promoTemplate.getApplicationArea()) {
            log.error("Only per user promos can be activated via promocodes. promo {} promocode {}",
                    promoTemplate, getPromoCodeData().getCode());
            throw new IllegalStateException("Illegal promo configuration. Only per-user promos can be activated via " +
                    "promocodes. promo id is " + promoTemplate);
        }

        if (PromoApplicationType.ONE_TIME == promoTemplate.getApplicationType()) {
            Option<UserPromoEntity> userPromoEntity = userPromoDao.findUserPromo(uid, promoTemplate.getId());

            if (userPromoEntity.isPresent()) {
                log.warn("One-time promo {} has already been activated {} for user {}. promo code {} can not be " +
                                "activated",
                        promoTemplate, userPromoEntity, uid, getPromoCodeData().getCode());
                return false;
            }
        }
        boolean result = promoTemplate.isActive();
        if (!result) {
            log.warn("Promo template {} is not active based on from/to dates. can not activate promo code {} ",
                    promoTemplate, getPromoCodeData().getCode());
            return false;
        }
        log.debug("Promo code {} is ok to proceed based on promo application type and dates {}",
                getPromoCodeData().getCode(),
                promoTemplate
        );
        return true;
    }

    @Override
    protected void activateImpl() {
        //activate if used = false is ok. multiple time promos don't get used
        PromoTemplate promoTemplate = promoTemplateSupplier.apply();
        promoTemplate.activatePromoForUser(uid, false, false);

        UserPromoEntity entity = userPromoDao.findUserPromo(uid, promoTemplate.getId())
                .orElseThrow(() -> {
                    log.error("Activation promo failed. promoCode {}, promoTemplate {}, uid {}",
                            getPromoCodeData().getCode(),
                            promoTemplate,
                            uid);
                    return new IllegalStateException("Activation promo failed");
                });

        userPromoCodeDao.create(
                UserPromoCodeDao.InsertData.builder()
                        .code(getPromoCodeData().getCode())
                        .uid(uid)
                        .userPromoId(Option.of(entity.getId()))
                        .build()
        );
    }

    @Override
    public String logUsedFor() {
        return uid.toString();
    }

    @Override
    public String getPromoCodeActivationType() {
        return "promo_template";
    }
}
