package ru.yandex.chemodan.app.psbilling.core.promocodes.rule.impl;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupProductDao;
import ru.yandex.chemodan.app.psbilling.core.dao.groups.GroupServiceDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.PromoTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.Group;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupProductEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.groups.GroupService;
import ru.yandex.chemodan.app.psbilling.core.entities.promocodes.PromoCodeType;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeData;
import ru.yandex.chemodan.app.psbilling.core.promocodes.model.PromoCodeFailCodeEnum;
import ru.yandex.chemodan.app.psbilling.core.promocodes.rule.PromoCodeRuleChecker;
import ru.yandex.chemodan.app.psbilling.core.promocodes.rule.PromoCodeRuleCheckerResult;
import ru.yandex.chemodan.app.psbilling.core.promocodes.rule.PromoCodeRuleContext;
import ru.yandex.chemodan.app.psbilling.core.synchronization.engine.Target;
import ru.yandex.inside.passport.PassportUid;

/**
 * Правило проверяет что текущие продукты по стоимости меньше или равны максимальному предложению из линеек
 */
@Slf4j
@RequiredArgsConstructor
public class HaveDiscountOnMoreExpensiveGroupProductPromoCodeRuleChecker implements PromoCodeRuleChecker {

    private final GroupServiceDao groupServiceDao;
    private final GroupProductDao groupProductDao;
    private final PromoTemplateDao promoTemplateDao;

    @Override
    public PromoCodeRuleCheckerResult check(
            PromoCodeData promoCodeData,
            Option<PassportUid> uidO,
            Option<Group> groupO,
            PromoCodeRuleContext context
    ) {
        log.debug("PromoCodeRuleChecker [HaveDiscountOnMoreExpensiveGroupProduct] check for promoCodeData={} uidO={} " +
                "groupO={} context={}", promoCodeData, uidO, groupO, context);

        if (promoCodeData.getType() != PromoCodeType.B2B) {
            throw new IllegalArgumentException("Rule checker only for b2b promo code");
        }

        Group group = groupO
                .orElseThrow(() -> new IllegalArgumentException("Group is required"));

        SetF<UUID> promoProductLineIds = promoCodeData.getPromoTemplateId()
                .map(p -> getPromoProductLines(context, p))
                .orElseThrow(() -> new IllegalArgumentException("Promo template id is required"));

        if (promoProductLineIds.isEmpty()) {
            throw new IllegalStateException("Promo product line is empty");
        }

        boolean haveCurrentExpensiveProduct = getGroupService(context, group)
                .stream()
                .filter(gs -> !gs.isHidden())
                .map(gs -> getGroupProduct(context, gs))
                .map(GroupProductEntity::getPricePerUserInMonth)
                .anyMatch(price -> checkPriceInPromoProductLines(context, promoProductLineIds, price));

        return haveCurrentExpensiveProduct
                ? PromoCodeRuleCheckerResult.fail(PromoCodeFailCodeEnum.EXPENSIVE_CURRENT_PRODUCT)
                : PromoCodeRuleCheckerResult.success();
    }

    private boolean checkPriceInPromoProductLines(
            PromoCodeRuleContext context,
            Collection<UUID> promoProductLineIds,
            BigDecimal price
    ) {
        return promoProductLineIds.stream()
                .flatMap(productLineId -> getProductByProductLine(context, productLineId).stream())
                .filter(gp -> gp.getTrialDefinitionId().isPresent())
                .map(GroupProductEntity::getPricePerUserInMonth)
                .max(BigDecimal::compareTo)
                .map(maxPrice -> maxPrice.compareTo(price) < 0)
                .orElse(false);
    }

    private ListF<GroupProductEntity> getProductByProductLine(PromoCodeRuleContext context, UUID productLineId) {
        return context.getCachedValue(
                "product_line." + productLineId + ".products",
                () -> groupProductDao.findByProductLine(productLineId)
        );
    }

    private SetF<UUID> getPromoProductLines(PromoCodeRuleContext context, UUID p) {
        return context.getCachedValue(
                "promo_template." + p + ".product_lines",
                () -> promoTemplateDao.getProductLines(p)
        );
    }

    private GroupProductEntity getGroupProduct(PromoCodeRuleContext context, GroupService gs) {
        return context.getCachedValue(
                "group_service." + gs.getId() + ".group_product",
                () -> groupProductDao.findByGroupServiceId(gs.getId())
        );
    }

    private ListF<GroupService> getGroupService(PromoCodeRuleContext context, Group group) {
        return context.getCachedValue(
                "group." + group.getId() + ".group_services",
                () -> groupServiceDao.find(group.getId(), Target.ENABLED)
        );
    }
}
