package ru.yandex.chemodan.app.psbilling.core.promos;

import java.util.UUID;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductLineDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.PromoTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.UserPromoDao;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationArea;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoStatusType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.UserPromoEntity;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.date.DateTimeUtils;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.DefaultObject;

public abstract class AbstractPromoTemplate extends DefaultObject implements PromoTemplate {
    protected final PromoTemplateEntity promoTemplate;
    protected final UserPromoDao userPromoDao;
    protected final TextsManager textsManager;
    protected final PromoPayloadParser promoPayloadParser;
    protected final ProductLineDao productLineDao;
    private final UserTimezoneHelper userTimezoneHelper;


    private final Function0<SetF<UUID>> productLinesProvider;
    private final MapF<PassportUid, UserPromoEntity> userPromoCache = Cf.concurrentHashMap();

    public AbstractPromoTemplate(PromoTemplateEntity promoTemplate, UserPromoDao userPromoDao,
                                 TextsManager textsManager,
                                 UserTimezoneHelper userTimezoneHelper, SetF<UUID> productLineIds,
                                 Option<UserPromoEntity> userPromoEntity, PromoPayloadParser promoPayloadParser,
                                 ProductLineDao productLineDao) {
        this(promoTemplate, userTimezoneHelper, userPromoDao, textsManager, promoPayloadParser, productLineDao,
                () -> productLineIds);
        userPromoEntity.ifPresent(e -> userPromoCache.put(e.getUid(), e));
    }

    public AbstractPromoTemplate(PromoTemplateEntity promoTemplate, UserTimezoneHelper userTimezoneHelper,
                                 UserPromoDao userPromoDao,
                                 TextsManager textsManager,
                                 PromoTemplateDao promoTemplateDao, PromoPayloadParser promoPayloadParser,
                                 ProductLineDao productLineDao) {
        this(promoTemplate, userTimezoneHelper, userPromoDao, textsManager,
                promoPayloadParser, productLineDao, () -> promoTemplateDao.getProductLines(promoTemplate.getId()));
    }

    private AbstractPromoTemplate(PromoTemplateEntity promoTemplate, UserTimezoneHelper userTimezoneHelper,
                                  UserPromoDao userPromoDao,
                                  TextsManager textsManager,
                                  PromoPayloadParser promoPayloadParser,
                                  ProductLineDao productLineDao, Function0<SetF<UUID>> productLinesProvider) {
        this.promoTemplate = promoTemplate;
        this.userPromoDao = userPromoDao;
        this.textsManager = textsManager;
        this.promoPayloadParser = promoPayloadParser;
        this.productLinesProvider = productLinesProvider.memoize();
        this.userTimezoneHelper = userTimezoneHelper;
        this.productLineDao = productLineDao;
    }

    Option<UserPromoEntity> getUserPromoEntity(PassportUid uid) {
        Option<UserPromoEntity> userPromoEntity = userPromoCache.getO(uid);
        if (!userPromoEntity.isPresent()) {
            userPromoEntity = lookupUserPromoEntity(uid);
        }
        return userPromoEntity;
    }

    private Option<UserPromoEntity> lookupUserPromoEntity(PassportUid uid) {
        Option<UserPromoEntity> userPromoEntity = userPromoDao.findUserPromo(uid, promoTemplate.getId());
        userPromoEntity.ifPresent(p -> userPromoCache.put(uid, p));
        return userPromoEntity;
    }


    @Override
    public SetF<UUID> getProductLineIds() {
        return productLinesProvider.apply();
    }

    @Override
    public boolean isExpired() {
        return promoTemplate.getToDate().map(Instant::isBeforeNow).orElse(false);
    }

    @Override
    public String getCode() {
        return promoTemplate.getCode();
    }

    @Override
    public Option<String> getTitle(String language) {
        return promoTemplate.getPromoNameTankerKey().flatMapO(key ->
                textsManager.findTranslation(key).findLanguageTranslation(language));
    }

    @Override
    public boolean isActive() {
        return promoTemplate.isActive();
    }

    public boolean isActiveInFuture() {
        return promoTemplate.isActiveInFuture();
    }

    @Override
    public Option<Instant> getToDate() {
        return promoTemplate.getToDate();
    }

    @Override
    public UUID getId() {
        return promoTemplate.getId();
    }

    public PromoApplicationType getApplicationType() {
        return promoTemplate.getApplicationType();
    }

    public PromoApplicationArea getApplicationArea() {
        return promoTemplate.getApplicationArea();
    }

    protected UserPromoEntity createUserPromo(PassportUid uid, PromoStatusType promoStatusType) {
        Instant from = getUserPromoFromDate();
        Option<Instant> to = getUserPromoToDate(uid, from);

        return userPromoDao.createOrUpdate(
                UserPromoDao.InsertData.builder()
                        .promoTemplateId(promoTemplate.getId())
                        .uid(uid)
                        .fromDate(from)
                        .toDate(to)
                        .promoStatusType(promoStatusType)
                        .build());
    }

    private Instant getUserPromoFromDate() {
        Instant from = promoTemplate.getFromDate();
        if (from.isBeforeNow()) {
            return Instant.now();
        }
        return from;
    }

    private Option<Instant> getUserPromoToDate(PassportUid uid, Instant userPromoFromDate) {
        Option<Instant> to = promoTemplate.getToDate();
        Option<CustomPeriod> durationO = promoTemplate.getDuration();
        if (durationO.isPresent()) {
            Instant possibleTo = userPromoFromDate.plus(durationO.get().toDurationFrom(userPromoFromDate));
            to = to.isPresent() && possibleTo.isAfter(to.get()) ? to : Option.of(possibleTo);
        }
        if (to.isPresent()) {
            DateTimeZone userTimezone = userTimezoneHelper.getUserTimezone(uid);
            return Option.of(DateTimeUtils.ceilTimeForTimezone(to.get(), userTimezone));
        }
        return to;
    }

}
