package ru.yandex.chemodan.app.psbilling.core.promos;

import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductLineDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.PromoTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.UserPromoDao;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoStatusType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.UserPromoEntity;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

class GlobalPromoTemplate extends AbstractPromoTemplate {
    private static final Logger logger = LoggerFactory.getLogger(GlobalPromoTemplate.class);

    public GlobalPromoTemplate(PromoTemplateEntity promoTemplate, UserPromoDao userPromoDao,
                               TextsManager textsManager, SetF<UUID> productLineIds,
                               Option<UserPromoEntity> userPromoEntity, UserTimezoneHelper userTimezoneHelper,
                               PromoPayloadParser promoPayloadParser, ProductLineDao productLineDao) {
        super(promoTemplate, userPromoDao, textsManager, userTimezoneHelper, productLineIds, userPromoEntity,
                promoPayloadParser, productLineDao);
    }

    public GlobalPromoTemplate(PromoTemplateEntity promoTemplate, UserPromoDao userPromoDao, TextsManager textsManager,
                               PromoTemplateDao promoTemplateDao, UserTimezoneHelper userTimezoneHelper,
                               PromoPayloadParser promoPayloadParser, ProductLineDao productLineDao) {
        super(promoTemplate, userTimezoneHelper, userPromoDao, textsManager, promoTemplateDao, promoPayloadParser,
                productLineDao);
    }

    @Override
    public boolean activatePromoForUser(PassportUid uid, boolean activateIfUsed, boolean sendEmail,
                                        Function<MailContext.MailContextBuilder, MailContext.MailContextBuilder> mailContextCustomizer) {
        throw new BadRequestException("Unable to activate global promo " + getCode() + " for uid " + uid);
    }

    @Override
    public boolean canActivateForUid(PassportUid uid, boolean activateIfUsed) {
        return false;
    }

    @Override
    public Option<Instant> canBeUsedUntilDate(Option<PassportUid> uid) {
        if (!canBeUsedForUid(uid)) {
            throw new IllegalStateException("cannot be used " + this);
        }

        return getToDate();
    }

    @Override
    public boolean canBeUsedForUid(Option<PassportUid> uidO) {
        if (getApplicationType() == PromoApplicationType.MULTIPLE_TIME || !uidO.isPresent()) {
            // если акция мультиактивизационная или у нас нет uid для проверки состояния по пользователю, то смотрим
            // доступна ли сама акция
            return isActive();
        }

        PassportUid uid = uidO.get();

        // у глобальных акций привязка к пользователю говорит только о том, что она уже использовалась
        // поэтому если привязки нет, то опять же смотрим на статус доступности самой акции
        return getUserPromoEntity(uid).map(UserPromoEntity::isActive).orElseGet(this::isActive);
    }

    @Override
    public boolean canBeUsedForUidInFuture(Option<PassportUid> uidO) {
        if (getApplicationType() == PromoApplicationType.MULTIPLE_TIME || !uidO.isPresent()) {
            // если акция мультиактивизационная или у нас нет uid для проверки состояния по пользователю, то смотрим
            // доступна ли сама акция
            return isActiveInFuture();
        }

        PassportUid uid = uidO.get();

        // у глобальных акций привязка к пользователю говорит только о том, что она уже использовалась
        // поэтому если привязки нет, то опять же смотрим на статус доступности самой акции
        return getUserPromoEntity(uid).map(UserPromoEntity::isActiveInFuture).orElseGet(this::isActiveInFuture);
    }

    @Override
    public void markUsedForUser(PassportUid uid) {
        Option<UserPromoEntity> userPromoEntity = getUserPromoEntity(uid);
        Validate.none(userPromoEntity);

        logger.info("set global promo {} used for user {}", getId(), uid);
        createUserPromo(uid, PromoStatusType.USED);
    }
}
