package ru.yandex.chemodan.app.psbilling.core.promos;

import java.util.UUID;

import lombok.EqualsAndHashCode;
import lombok.ToString;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function0;
import ru.yandex.chemodan.app.psbilling.core.entities.CustomPeriod;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationArea;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoApplicationType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoTemplateEntity;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.date.DateTimeUtils;
import ru.yandex.inside.passport.PassportUid;

@ToString(onlyExplicitlyIncluded = true)
@EqualsAndHashCode(onlyExplicitlyIncluded = true)
public abstract class NewAbstractPromoTemplate<Usage, Entity> implements NewPromoTemplate<Usage, Entity> {

    @ToString.Include
    @EqualsAndHashCode.Include
    protected final PromoTemplateEntity promoTemplate;
    protected final UserTimezoneHelper userTimezoneHelper;
    private final Function0<SetF<UUID>> productLinesProvider;
    protected final MapF<Usage, Entity> promoEntityCache = Cf.concurrentHashMap();

    public NewAbstractPromoTemplate(
            PromoTemplateEntity promoTemplate,
            UserTimezoneHelper userTimezoneHelper,
            Function0<SetF<UUID>> productLinesProvider
    ) {
        this.promoTemplate = promoTemplate;
        this.userTimezoneHelper = userTimezoneHelper;
        this.productLinesProvider = productLinesProvider.memoize();
    }

    protected Option<Entity> getPromoEntity(Usage usage) {
        Option<Entity> entity = promoEntityCache.getO(usage);
        if (!entity.isPresent()) {
            entity = lookupPromoEntity(usage);
            entity.ifPresent(p -> promoEntityCache.put(usage, p));
        }
        return entity;
    }

    public abstract Option<Entity> lookupPromoEntity(Usage usage);

    @Override
    public SetF<UUID> getProductLineIds() {
        return productLinesProvider.apply();
    }

    @Override
    public boolean isExpired() {
        return promoTemplate.getToDate().map(Instant::isBeforeNow).orElse(false);
    }

    @Override
    public String getCode() {
        return promoTemplate.getCode();
    }

    @Override
    public Option<UUID> getPromoNameTankerKey() {
        return promoTemplate.getPromoNameTankerKey();
    }

    @Override
    public boolean isActive() {
        return promoTemplate.isActive();
    }

    @Override
    public Option<Instant> getToDate() {
        return promoTemplate.getToDate();
    }

    @Override
    public UUID getId() {
        return promoTemplate.getId();
    }

    public PromoApplicationType getApplicationType() {
        return promoTemplate.getApplicationType();
    }

    @Override
    public PromoApplicationArea getApplicationArea() {
        return promoTemplate.getApplicationArea();
    }
    protected Instant getPromoFromDate() {
        Instant from = promoTemplate.getFromDate();
        if (from.isBeforeNow()) {
            return Instant.now();
        }
        return from;
    }

    protected Option<Instant> calcPromoToDate(PassportUid uid, Instant promoFromDate) {
        Option<Instant> to = promoTemplate.getToDate();
        Option<CustomPeriod> durationO = promoTemplate.getDuration();
        if (durationO.isPresent()) {
            Instant possibleTo = promoFromDate.plus(durationO.get().toDurationFrom(promoFromDate));
            to = to.isPresent() && possibleTo.isAfter(to.get()) ? to : Option.of(possibleTo);
        }
        if (to.isPresent()) {
            DateTimeZone userTimezone = userTimezoneHelper.getUserTimezone(uid);
            return Option.of(DateTimeUtils.ceilTimeForTimezone(to.get(), userTimezone));
        }
        return to;
    }

}
