package ru.yandex.chemodan.app.psbilling.core.promos;

import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.psbilling.core.dao.products.ProductLineDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.PromoTemplateDao;
import ru.yandex.chemodan.app.psbilling.core.dao.promos.UserPromoDao;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoStatusType;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.PromoTemplateEntity;
import ru.yandex.chemodan.app.psbilling.core.entities.promos.UserPromoEntity;
import ru.yandex.chemodan.app.psbilling.core.mail.MailContext;
import ru.yandex.chemodan.app.psbilling.core.tasks.execution.TaskScheduler;
import ru.yandex.chemodan.app.psbilling.core.texts.TextsManager;
import ru.yandex.chemodan.util.blackbox.UserTimezoneHelper;
import ru.yandex.chemodan.util.exception.BadRequestException;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


class PerUserPromoTemplate extends AbstractPromoTemplate {
    private static final Logger logger = LoggerFactory.getLogger(PerUserPromoTemplate.class);
    private final TaskScheduler taskScheduler;

    public PerUserPromoTemplate(PromoTemplateEntity promoTemplate, SetF<UUID> productLineIds,
                                Option<UserPromoEntity> userPromoEntity,
                                TaskScheduler taskScheduler, UserPromoDao userPromoDao, TextsManager textsManager,
                                UserTimezoneHelper userTimezoneHelper, PromoPayloadParser promoPayloadParser,
                                ProductLineDao productLineDao) {
        super(promoTemplate, userPromoDao, textsManager, userTimezoneHelper, productLineIds, userPromoEntity,
                promoPayloadParser, productLineDao);
        this.taskScheduler = taskScheduler;
    }

    public PerUserPromoTemplate(PromoTemplateEntity promoTemplate, PromoTemplateDao promoTemplateDao,
                                TaskScheduler taskScheduler, UserPromoDao userPromoDao, TextsManager textsManager,
                                UserTimezoneHelper userTimezoneHelper, PromoPayloadParser promoPayloadParser,
                                ProductLineDao productLineDao) {
        super(promoTemplate, userTimezoneHelper, userPromoDao, textsManager, promoTemplateDao, promoPayloadParser,
                productLineDao);
        this.taskScheduler = taskScheduler;
    }

    @Override
    public boolean canBeUsedForUid(Option<PassportUid> uidO) {
        if (!uidO.isPresent()) {
            // если акция для пользователя и у нас нет uid для проверки состояния по пользователю,
            // то такую акцию нельзя использовать не активировав
            return false;
        }
        PassportUid uid = uidO.get();

        // user promo template can be used only if activated for user
        return getUserPromoEntity(uid).map(UserPromoEntity::isActive).orElse(false);
    }

    @Override
    public boolean canBeUsedForUidInFuture(Option<PassportUid> uidO) {
        if (!uidO.isPresent()) {
            // если акция для пользователя и у нас нет uid для проверки состояния по пользователю,
            // то такую акцию нельзя использовать не активировав
            return false;
        }
        PassportUid uid = uidO.get();

        // user promo template can be used only if activated for user
        return getUserPromoEntity(uid).map(UserPromoEntity::isActiveInFuture).orElse(false);
    }

    @Override
    public Option<Instant> canBeUsedUntilDate(Option<PassportUid> uidO) {
        if (!canBeUsedForUid(uidO)) {
            throw new IllegalStateException("cannot be used " + this);
        }
        PassportUid uid = uidO.get();
        UserPromoEntity userPromoEntity = getUserPromoEntity(uid)
                .getOrThrow("if promo template can be used by some user, then must be activated user promo");

        return userPromoEntity.getToDate();
    }

    @Override
    public boolean canActivateForUid(PassportUid uid, boolean activateIfUsed) {
        if (isExpired()) {
            logger.info("can't activate {} promo for user {} cause it's expired", promoTemplate, uid);
            return false;
        }

        return canActivateUserPromo(uid, activateIfUsed);
    }

    @Override
    public boolean activatePromoForUser(PassportUid uid, boolean activateIfUsed, boolean sendEmail,
                                        Function<MailContext.MailContextBuilder, MailContext.MailContextBuilder> mailContextCustomizer) {
        if (isExpired()) {
            throw new BadRequestException("promo with key " + getCode() + " is expired");
        }

        if (!canActivateUserPromo(uid, activateIfUsed)) {
            return false;
        }

        Option<UserPromoEntity> userPromoO = getUserPromoEntity(uid);
        // защита от спама письмам пользователей, если вдруг кто-то будет часто дергать активацию акции
        if (sendEmail && userPromoO.isPresent() && userPromoO.get().isActive()) {
            logger.info("sendEmail was changed to false cause reactivating already active promo {}", userPromoO);
            sendEmail = false;
        }

        UserPromoEntity userPromo = createUserPromo(uid, PromoStatusType.ACTIVE);
        if (sendEmail) {
            sendActivationEmail(uid, userPromo.getId(), mailContextCustomizer);
        }
        return true;
    }

    protected void sendActivationEmail(PassportUid uid, UUID userPromoId,
                                       Function<MailContext.MailContextBuilder, MailContext.MailContextBuilder> mailContextCustomizer) {

        if (!promoTemplate.getActivationEmailTemplate().isPresent()) {
            logger.warn("can't send email of promo [{}] cause email template has not been set", promoTemplate);
            return;
        }
        String activationEmailTemplate = promoTemplate.getActivationEmailTemplate().get();

        MailContext.MailContextBuilder contextBuilder = MailContext.builder().to(uid)
                .promoId(Option.of(promoTemplate.getId()))
                .userPromoId(Option.of(userPromoId));
        contextBuilder = mailContextCustomizer.apply(contextBuilder);
        taskScheduler.schedulePromoEmailTask(activationEmailTemplate, contextBuilder.build());
    }

    @Override
    public void markUsedForUser(PassportUid uid) {
        Option<UserPromoEntity> userPromoEntityO = getUserPromoEntity(uid);
        Validate.some(userPromoEntityO);
        UserPromoEntity userPromoEntity = userPromoEntityO.single();
        Validate.isTrue(userPromoEntity.isActive());

        logger.info("set user promo {} used for user {}", getId(), uid);
        userPromoDao.setStatus(userPromoEntity.getId(), PromoStatusType.USED);
    }

    private boolean canActivateUserPromo(PassportUid uid, boolean activateIfUsed) {
        Option<UserPromoEntity> userPromoEntity = getUserPromoEntity(uid);

        if (!userPromoEntity.isPresent() ||
                (!userPromoEntity.get().isActive()
                        && userPromoEntity.get().getStatus() == PromoStatusType.ACTIVE) //Not used expired promo
        ) {
            logger.info("can activate {} promo for user {}. found userPromo: {userPromoEntity}",
                    promoTemplate, uid, userPromoEntity);
            return true;
        }

        boolean result = userPromoEntity.get().getStatus() == PromoStatusType.USED && activateIfUsed;
        logger.info("can activate = {} for {} promo for user {}. found userPromo: {userPromoEntity}",
                result, promoTemplate, uid, userPromoEntity);
        return result;
    }
}
